# AI Function Calling Implementation

## Overview
Smart Friend now supports AI function calling, allowing the assistant to perform real actions like setting reminders, checking weather, and searching the web.

## Available Functions

### 1. Set Reminder (`set_reminder`)
Creates a reminder for the user with a title, optional description, and scheduled time.

**Example prompts:**
- "Set a reminder for my meeting tomorrow at 10am"
- "Remind me to call mom at 5pm"
- "আগামীকাল সকাল ১০টায় মিটিং এর জন্য একটি রিমাইন্ডার সেট করুন"

**Parameters:**
- `title` (required): The reminder title
- `description` (optional): Additional details
- `remind_at` (required): ISO 8601 datetime string

### 2. Get Weather (`get_weather`)
Retrieves current weather information for any location worldwide.

**Example prompts:**
- "What's the weather like in Dhaka?"
- "How's the weather in London today?"
- "ঢাকার আবহাওয়া কেমন?"

**Parameters:**
- `location` (required): City name or location

**Returns:**
- Current temperature, conditions, humidity, wind speed
- Feels-like temperature, UV index
- Today's high/low forecast

### 3. Search Web (`search_web`)
Searches the web for current information on any topic.

**Example prompts:**
- "Search for latest news about Bangladesh"
- "Find information about React 19 features"
- "বাংলাদেশের সর্বশেষ খবর খুঁজুন"

**Parameters:**
- `query` (required): The search query

**Returns:**
- Relevant search results with titles, snippets, and URLs

## How It Works

### Architecture Flow

1. **User sends message** → Chat interface
2. **AI processes** → Lovable AI Gateway with function definitions
3. **Function call detected** → Streaming response includes `tool_calls`
4. **Execute function** → `execute-function` edge function
5. **Results returned** → Sent back to AI with context
6. **Final response** → AI generates user-friendly response with results

### Technical Implementation

#### Edge Functions

**`smart-friend-chat`**
- Handles chat requests with function definitions
- Streams responses from Lovable AI
- Returns both content and tool_calls

**`execute-function`**
- Executes requested functions securely
- Validates user authentication
- Returns structured results

#### Frontend (Chat.tsx)

The chat interface handles the full function calling flow:

```typescript
// 1. Parse tool calls from streaming response
if (delta?.tool_calls) {
  // Accumulate function arguments
}

// 2. Execute functions when complete
const toolResults = await Promise.all(
  toolCalls.map(async (toolCall) => {
    const result = await supabase.functions.invoke("execute-function", {
      body: {
        functionName: toolCall.function.name,
        arguments: JSON.parse(toolCall.function.arguments),
      },
    });
    return result;
  })
);

// 3. Send results back to AI
const finalResponse = await fetch(chatUrl, {
  body: JSON.stringify({
    messages: [...history, toolResults],
  }),
});
```

## UI Components

### FunctionCallIndicator
Displays visual feedback when functions are executing:
- Calendar icon for reminders
- Cloud icon for weather
- Search icon for web searches
- Animated spinner during execution

### FunctionExamples
Shows example prompts when chat is empty:
- Quick-start suggestions
- Bilingual support (English/Bangla)
- Click to populate input

## Security

- **Authentication required**: All function calls require valid user session
- **Server-side execution**: Functions execute in secure edge functions
- **Rate limiting**: Protected by Lovable AI rate limits
- **Input validation**: Parameters validated before execution

## Error Handling

Functions gracefully handle errors:
- Network failures
- Invalid parameters
- API rate limits
- Service unavailability

Errors are returned to the AI, which explains them to the user in natural language.

## Adding New Functions

To add a new function:

1. **Define tool in `smart-friend-chat/index.ts`:**
```typescript
{
  type: "function",
  function: {
    name: "my_function",
    description: "What this function does",
    parameters: {
      type: "object",
      properties: {
        param1: { type: "string", description: "Parameter description" }
      },
      required: ["param1"]
    }
  }
}
```

2. **Implement in `execute-function/index.ts`:**
```typescript
case "my_function":
  result = await myFunction(args.param1);
  break;
```

3. **Add UI indicator (optional):**
Update `FunctionCallIndicator.tsx` with icon and label.

## Testing

Test function calling with these prompts:

**Reminders:**
- "Remind me to submit the report by Friday 5pm"
- "Set a reminder for gym tomorrow at 6am"

**Weather:**
- "What's the weather in New York?"
- "ঢাকার আবহাওয়া দেখান"

**Search:**
- "Search for TypeScript 5.0 features"
- "বাংলাদেশের জনসংখ্যা কত?"

## Monitoring

Monitor function usage:
- Check edge function logs in Lovable Cloud
- View reminder creation in database
- Track API usage and rate limits

## Future Enhancements

Potential additions:
- Translation function
- Calendar integration
- Task management
- Email sending
- Payment processing
- Smart home control
