# Phase 7 Extension: Auto Composer + Vendor Builder + Laravel Runtime Setup

## Overview

This extension to Phase 7 (Auto DevOps Engine) adds comprehensive Laravel-specific capabilities including automatic Composer dependency management, artisan command execution, and full Laravel runtime setup for deployments.

## Features Implemented

### 1. Laravel Runtime Configuration

**Location**: `src/lib/devops/laravelRuntimeConfig.ts`

Central configuration module for Laravel deployment settings:

- **PHP Version Selection**: Choose between PHP 8.1, 8.2, or 8.3
- **Composer Options**:
  - Force composer install (rebuild vendor)
  - Composer update mode
  - No-dev flag support
  - Autoloader optimization
- **Artisan Command Configuration**:
  - Toggle migrations (on/off)
  - Toggle database seeds (on/off)
  - Toggle optimization (on/off)
- **Timeout Settings**: Default 600 seconds for long-running operations

Configuration is persisted in localStorage and can be updated via the DevOps UI.

### 2. Remote Command Runner

**Location**: `src/lib/devops/remoteCommandRunner.ts`

Unified command execution system supporting multiple deployment targets:

**Supported Methods**:
- **SSH**: Direct command execution via SSH protocol
- **cPanel**: Terminal API integration for cPanel hosting
- **HTTP**: Safe HTTP-based artisan runner (whitelist-protected)

**Features**:
- Stdout/stderr capture
- Exit code tracking
- Duration measurement
- Comprehensive logging
- Security whitelisting (artisan commands only)

**Whitelisted Artisan Commands**:
- `key:generate`
- `migrate`
- `db:seed`
- `optimize`
- `config:cache`
- `route:cache`
- `view:cache`
- `cache:clear`
- `config:clear`
- `route:clear`
- `view:clear`

### 3. Auto Composer Setup

**Location**: `src/hooks/useComposerSetup.tsx`

Automated Composer dependency management:

**Capabilities**:
- Detect composer.json presence
- Smart install (only if vendor missing)
- Force reinstall option
- Update mode for existing vendor
- No-dev flag support
- Autoloader optimization
- Real-time progress logging
- Duration tracking
- Error handling with suggestions

**Usage**:
```typescript
const { runComposerSetup, installing, logs, lastResult } = useComposerSetup();

await runComposerSetup(target, config, {
  forceReinstall: true,
  optimizeAutoloader: true
});
```

### 4. Laravel Post-Deploy Runner

**Location**: `src/hooks/useLaravelPostDeploy.tsx`

Automated artisan command execution after deployment:

**Default Pipeline**:
1. `php artisan key:generate --force`
2. `php artisan migrate --force` (if enabled)
3. `php artisan db:seed --force` (if enabled)
4. `php artisan config:cache` (if optimize enabled)
5. `php artisan route:cache` (if optimize enabled)
6. `php artisan view:cache` (if optimize enabled)
7. `php artisan optimize` (if optimize enabled)

**Features**:
- Sequential command execution
- Per-command result tracking
- Continue on error (mark as "needs attention")
- Real-time logging
- Duration tracking per command
- Single command execution mode

**Usage**:
```typescript
const { runLaravelPostDeploy, running, logs, results } = useLaravelPostDeploy();

await runLaravelPostDeploy(target, config);

// Or run single command
await runSingleArtisanCommand(target, 'php artisan cache:clear');
```

### 5. DevOps UI Integration

**Location**: `src/pages/StudioDevOps.tsx`

New "Laravel Runtime" tab in Auto DevOps Engine:

**UI Components**:
- **PHP Version Selector**: Dropdown for PHP 8.1/8.2/8.3
- **Composer Options**: Checkbox for force reinstall
- **Artisan Toggles**:
  - Run migrations
  - Run seeds
  - Run optimization
- **Action Buttons**:
  - "Run Composer Install"
  - "Run Artisan Commands"
- **Real-time Logs**:
  - Composer output panel
  - Artisan command results (with success/fail indicators)
  - Detailed logs panel

**Visual Features**:
- Success/error color coding (green/red borders)
- Loading spinners during operations
- Duration display for each command
- Success/failure icons (CheckCircle/XCircle)

### 6. Deployment Pipeline Integration

**Updated**: `src/hooks/useDeployment.tsx`

Laravel runtime steps integrated into main deployment pipeline:

**Enhanced Pipeline Order**:
1. Build project
2. Create deployment package
3. Upload to server
4. Extract files
5. **Run Composer install** (new)
6. **Run Laravel artisan commands** (new)
7. Configure environment variables
8. Install SSL (if enabled)
9. Mark deployment complete

**Logs Include**:
- "Installing Composer dependencies..."
- "✓ Composer install completed"
- "Running Laravel artisan commands..."
- "✓ php artisan key:generate --force"
- "✓ php artisan migrate --force"
- "✓ php artisan optimize"

### 7. Error Handling & Integration

**Auto Bug Finder Integration**:
- Composer failures create error entries in `/studio/errors`
- Artisan failures marked as "needs attention"
- Full stdout/stderr captured in logs
- Suggested fixes provided:
  - Change PHP version
  - Check database connection
  - Verify composer.lock conflicts
  - Adjust memory limits

**Self-Repair Hooks**:
- Retry composer with different flags
- Skip seeds and retry
- Re-run individual artisan commands
- Adjust timeout settings

## Security & Safety

### Restrictions
- ✅ Only Owner (Tasnim) can configure SSH/cPanel credentials
- ✅ No arbitrary shell commands allowed
- ✅ Only whitelisted artisan commands permitted
- ✅ HTTP runner uses command whitelist
- ✅ All operations logged for audit trail

### Safe HTTP Runner
If using HTTP-based execution (fallback), a small `artisan-runner.php` script is deployed with strict whitelisting:
- Accepts only pre-approved artisan commands
- No shell command injection possible
- Returns structured JSON responses
- Logs all execution attempts

## Usage Guide

### Basic Laravel Deployment Flow

1. **Configure Runtime Settings**:
   - Go to Studio → DevOps → Laravel Runtime tab
   - Select PHP version (8.2 recommended)
   - Enable desired artisan commands
   - Toggle composer force install if needed

2. **Deploy Project**:
   - Use any deployment method (FTP/SSH/cPanel)
   - Pipeline automatically runs:
     - Composer install
     - Artisan commands based on config

3. **Monitor Execution**:
   - Watch real-time logs in Composer/Artisan output panels
   - Check command results (green = success, red = failed)
   - Review duration metrics

4. **Handle Errors**:
   - Check `/studio/errors` for auto-generated error reports
   - Review suggested fixes
   - Adjust configuration and retry

### Manual Execution

You can run Composer/Artisan commands manually:

```typescript
// In StudioDevOps page:
// 1. Click "Run Composer Install" - installs dependencies
// 2. Click "Run Artisan Commands" - runs configured artisan commands
```

### Advanced Configuration

Edit Laravel runtime config programmatically:

```typescript
import { saveLaravelConfig } from "@/lib/devops/laravelRuntimeConfig";

saveLaravelConfig({
  phpVersion: "8.3",
  runMigrations: true,
  runSeed: false,
  runOptimize: true,
  forceComposerInstall: false
});
```

## Testing Checklist

✅ **Composer Install**:
- [ ] Fresh Laravel app deploys successfully
- [ ] Vendor folder created on server
- [ ] Dependencies installed correctly
- [ ] Logs show composer output

✅ **Artisan Commands**:
- [ ] Key generation succeeds
- [ ] Migrations run without errors
- [ ] Optimization caches created
- [ ] All commands logged properly

✅ **Error Handling**:
- [ ] Composer failures create error entries
- [ ] Artisan failures marked appropriately
- [ ] Logs capture stdout/stderr
- [ ] Suggested fixes appear

✅ **UI/UX**:
- [ ] Laravel Runtime tab displays correctly
- [ ] Configuration saves persist
- [ ] Real-time logs update during execution
- [ ] Success/fail indicators accurate

✅ **Integration**:
- [ ] Deployment pipeline includes Laravel steps
- [ ] No conflicts with existing Phase 7 features
- [ ] Works with FTP/SSH/cPanel methods
- [ ] Health checks pass after Laravel setup

## Future Enhancements

Potential additions for future phases:

1. **PHP Version Auto-Detection**: Automatically detect and recommend PHP version based on composer.json
2. **Composer Package Inspector**: UI to view installed packages and versions
3. **Artisan Command Builder**: GUI to construct custom artisan commands
4. **Queue Worker Setup**: Automated queue worker configuration
5. **Scheduler Setup**: Automated cron job setup for task scheduling
6. **Multi-Environment Support**: Different configs for staging/production
7. **Rollback Integration**: Restore previous vendor/composer state
8. **Performance Metrics**: Track artisan command execution times over deployments

## Notes

- This extension is **non-breaking** - all existing Phase 7 features remain intact
- Laravel runtime steps are **optional** - disable in config if not needed
- All operations are **logged** for debugging and audit purposes
- Security is prioritized - **no arbitrary command execution**
- Compatible with all deployment methods from Phase 8

## Related Documentation

- [PHASE_7_COMPLETION_REPORT.md](./PHASE_7_COMPLETION_REPORT.md) - Original Auto DevOps implementation
- [PHASE_8_COMPLETION_REPORT.md](./PHASE_8_COMPLETION_REPORT.md) - Self Deploy Engine
- [PHASE_E_COMPLETION_REPORT.md](./PHASE_E_COMPLETION_REPORT.md) - Backend AI Builder (Laravel)

---

**Status**: ✅ Complete and Operational  
**Last Updated**: 2025-01-19  
**Phase**: 7 Extension - Laravel Runtime
