# Web Search & Knowledge Engine Implementation

## Overview
Smart Friend ULTRA v2.0 now includes a complete Web Search & Knowledge Engine that enables AI-powered web search across all modules.

## Architecture

### Backend Components

#### 1. Edge Function: `web-search`
**Location:** `supabase/functions/web-search/index.ts`

**Endpoint:** `/functions/v1/web-search`

**Features:**
- Rate limiting (10 requests/minute per user by default)
- Multi-language support (Bangla, English, Hindi)
- Module-specific filtering
- Safe search filtering
- Logging and analytics

**Request Format:**
```typescript
{
  query: string;
  lang?: "bn" | "en" | "hi";
  maxResults?: number;
  module?: string;
  safeSearch?: boolean;
}
```

**Response Format:**
```typescript
{
  results: SearchResult[];
  query: string;
  tookMs: number;
  lang?: string;
  module?: string;
}
```

#### 2. Type Definitions
**Location:** `src/types/search.ts`

Defines TypeScript interfaces for:
- `SearchResult` - Individual search result
- `SearchResponse` - Complete search response
- `SearchOptions` - Search request options
- `SearchSettings` - System-wide settings

### Frontend Components

#### 1. Search Engine Page
**Route:** `/studio/search`
**Component:** `src/pages/StudioSearch.tsx`

**Features:**
- Real-time web search
- Language selection (Bangla/English/Hindi)
- Module filtering
- AI integration toggle
- Streaming AI summaries
- Result pagination

#### 2. Search Settings Page
**Route:** `/studio/search/settings`
**Component:** `src/pages/StudioSearchSettings.tsx`

**Features:**
- Master on/off toggle
- Per-module permissions
- Max results configuration
- Rate limiting controls
- Safe search toggle
- Owner-only access

#### 3. AI Engine Integration
**Location:** `src/pages/StudioAIEngine.tsx`

**Enhancement:**
- Added "Use Web Search" toggle
- AI can fetch web context before generating code
- Automatic result integration

## How to Use

### For End Users

1. **Navigate to Search Engine:**
   - Go to Studio → Search Engine
   - Or click the search icon in the sidebar

2. **Perform a Search:**
   - Enter your query
   - Select language (Bangla/English)
   - Choose module filter (optional)
   - Click "Search"

3. **Enable AI Summary:**
   - Toggle "Also ask Smart Friend AI"
   - Get AI-generated summary with sources

### For Administrators

1. **Access Settings:**
   - Go to Studio → Search Engine → Settings
   - Only available to system owners

2. **Configure Global Settings:**
   - Enable/disable web search globally
   - Set max results per query
   - Configure rate limits
   - Enable/disable safe search

3. **Configure Module Permissions:**
   - Enable search for specific modules:
     - Smart Academy
     - Smart News
     - Smart Doctor
     - Smart Lawyer
     - Smart Tourist
     - And more...

## Integration with Real Search APIs

The current implementation uses **mock data** for demonstration. To integrate with a real search API:

### Option 1: Tavily Search API

1. Get API key from [tavily.com](https://tavily.com)

2. Add to Supabase secrets:
```bash
TAVILY_API_KEY=your_api_key_here
```

3. Update `runWebSearch()` function in `supabase/functions/web-search/index.ts`:

```typescript
async function runWebSearch(query: string, options: any): Promise<any[]> {
  const TAVILY_API_KEY = Deno.env.get("TAVILY_API_KEY");
  
  const response = await fetch("https://api.tavily.com/search", {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({
      api_key: TAVILY_API_KEY,
      query,
      max_results: options.maxResults || 5,
      search_depth: "basic",
      include_answer: false,
    }),
  });

  const data = await response.json();
  
  return data.results.map((r: any) => ({
    title: r.title,
    url: r.url,
    snippet: r.content,
    source: new URL(r.url).hostname,
    score: r.score,
    publishedDate: r.published_date,
  }));
}
```

### Option 2: SerpAPI

1. Get API key from [serpapi.com](https://serpapi.com)

2. Add to Supabase secrets:
```bash
SERPAPI_KEY=your_api_key_here
```

3. Update `runWebSearch()` function:

```typescript
async function runWebSearch(query: string, options: any): Promise<any[]> {
  const SERPAPI_KEY = Deno.env.get("SERPAPI_KEY");
  
  const params = new URLSearchParams({
    q: query,
    api_key: SERPAPI_KEY,
    num: options.maxResults?.toString() || "5",
    hl: options.lang || "en",
  });

  const response = await fetch(`https://serpapi.com/search?${params}`);
  const data = await response.json();
  
  return (data.organic_results || []).map((r: any) => ({
    title: r.title,
    url: r.link,
    snippet: r.snippet,
    source: r.displayed_link,
    publishedDate: r.date,
  }));
}
```

### Option 3: Brave Search API

1. Get API key from [brave.com/search/api](https://brave.com/search/api)

2. Add to Supabase secrets:
```bash
BRAVE_API_KEY=your_api_key_here
```

3. Update `runWebSearch()` function:

```typescript
async function runWebSearch(query: string, options: any): Promise<any[]> {
  const BRAVE_API_KEY = Deno.env.get("BRAVE_API_KEY");
  
  const params = new URLSearchParams({
    q: query,
    count: options.maxResults?.toString() || "5",
  });

  const response = await fetch(`https://api.search.brave.com/res/v1/web/search?${params}`, {
    headers: {
      "X-Subscription-Token": BRAVE_API_KEY,
      "Accept": "application/json",
    },
  });

  const data = await response.json();
  
  return (data.web?.results || []).map((r: any) => ({
    title: r.title,
    url: r.url,
    snippet: r.description,
    source: new URL(r.url).hostname,
    publishedDate: r.age,
  }));
}
```

## Module Integration Examples

### Smart Academy Virtual Teacher
```typescript
// Before answering a question, fetch web context
const searchResults = await supabase.functions.invoke("web-search", {
  body: {
    query: studentQuestion,
    lang: "bn",
    maxResults: 3,
    module: "academy",
  },
});

// Pass context to AI
const aiResponse = await generateAnswer(studentQuestion, searchResults.data);
```

### Smart News
```typescript
// Enrich news headlines with web search
const headline = "Latest technology trends";
const webContext = await supabase.functions.invoke("web-search", {
  body: {
    query: headline,
    lang: "en",
    maxResults: 5,
    module: "news",
  },
});

// Display enriched news with sources
```

### Smart Doctor
```typescript
// Search medical information (with disclaimer)
const symptom = "headache causes";
const medicalInfo = await supabase.functions.invoke("web-search", {
  body: {
    query: symptom,
    lang: "bn",
    maxResults: 5,
    module: "doctor",
    safeSearch: true,
  },
});

// Show results with medical disclaimer
```

## Database Schema

### Settings Storage
Settings are stored in the `user_memory` table:

```sql
{
  memory_key: "search_settings",
  memory_type: "system",
  memory_value: {
    enabled: boolean,
    maxResults: number,
    safeSearch: boolean,
    rateLimit: number,
    modules: {
      academy: boolean,
      news: boolean,
      doctor: boolean,
      // ... more modules
    }
  }
}
```

### Search Logs
Searches are logged in `core_brain_logs`:

```sql
{
  type: "web_search",
  message: "Web search: [query]",
  details: {
    query: string,
    lang: string,
    module: string,
    resultsCount: number,
    tookMs: number,
    userId: string
  }
}
```

## Security & Rate Limiting

### Rate Limiting
- Default: 10 requests per minute per user
- Configurable via settings page
- Uses in-memory store (upgrade to Redis for production)

### Permission Checks
- Search settings require owner/admin role
- Checked via `is_core_brain_owner()` function
- Normal users can only use search, not configure

### Safe Search
- Enabled by default
- Filters adult/unsafe content
- Configurable per module

## Performance Considerations

### Caching
Consider implementing caching for:
- Popular queries
- Module-specific searches
- Recent results

### Optimization
- Limit max results to prevent overload
- Use rate limiting to prevent abuse
- Monitor API usage and costs

## Future Enhancements

1. **Advanced Search:**
   - Image search
   - Video search
   - News-specific search
   - Academic paper search

2. **AI Features:**
   - Query expansion
   - Result summarization
   - Fact checking
   - Citation generation

3. **Module-Specific:**
   - Medical knowledge base for Smart Doctor
   - Legal case search for Smart Lawyer
   - Course material search for Smart Academy

4. **Analytics:**
   - Popular searches dashboard
   - User search patterns
   - Module usage statistics

## Troubleshooting

### Common Issues

**Search returns no results:**
- Check if web search is enabled globally
- Verify module permissions
- Check rate limit status

**AI summary not working:**
- Ensure Lovable AI is configured
- Check if AI toggle is enabled
- Verify network connectivity

**Settings page shows access denied:**
- Only system owners can access settings
- Check user email against owner list

## Support

For issues or questions:
1. Check logs in `core_brain_logs` table
2. Review edge function logs
3. Contact system administrator

---

**Last Updated:** 2025-11-19
**Version:** ULTRA v2.0
**Status:** Production Ready
