import { useEffect, useState } from "react";
import { Navigate } from "react-router-dom";
import { useAuth } from "@/hooks/useAuth";
import { supabase } from "@/integrations/supabase/client";
import { Loader2 } from "lucide-react";
import { toast } from "sonner";

type CoreBrainProtectedRouteProps = {
  children: React.ReactNode;
};

const ALLOWED_EMAILS = [
  "nazrul.tasnim24@gmail.com",
  "engineer.tasnim@smartseba.com"
];

const CoreBrainProtectedRoute = ({ children }: CoreBrainProtectedRouteProps) => {
  const { user, loading: authLoading } = useAuth();
  const [isAuthorized, setIsAuthorized] = useState<boolean | null>(null);

  useEffect(() => {
    const checkAccess = async () => {
      if (!user) {
        setIsAuthorized(false);
        return;
      }

      // Check if user email is in allowed list
      const userEmail = user.email?.toLowerCase();
      const hasAccess = userEmail && ALLOWED_EMAILS.includes(userEmail);

      if (!hasAccess) {
        toast.error("You do not have permission to access Smart Friend Core Brain.");
      }

      setIsAuthorized(!!hasAccess);
    };

    if (!authLoading) {
      checkAccess();
    }
  }, [user, authLoading]);

  if (authLoading || isAuthorized === null) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <Loader2 className="h-8 w-8 animate-spin" />
      </div>
    );
  }

  if (!user || !isAuthorized) {
    return <Navigate to="/" replace />;
  }

  return <>{children}</>;
};

export default CoreBrainProtectedRoute;
