import { useEffect, useState } from "react";
import { supabase } from "@/integrations/supabase/client";
import { Card } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Brain, AlertCircle, CheckCircle2, AlertTriangle } from "lucide-react";
import { useNavigate } from "react-router-dom";

type SystemStatus = "healthy" | "warning" | "error";

const CoreBrainWidget = () => {
  const [status, setStatus] = useState<SystemStatus>("healthy");
  const [errorCount, setErrorCount] = useState(0);
  const [warningCount, setWarningCount] = useState(0);
  const [isAuthorized, setIsAuthorized] = useState(false);
  const navigate = useNavigate();

  useEffect(() => {
    checkAuthorization();
  }, []);

  const checkAuthorization = async () => {
    const { data: { user } } = await supabase.auth.getUser();
    if (!user) return;

    const authorized = user.email && [
      'nazrul.tasnim24@gmail.com',
      'engineer.tasnim@smartseba.com'
    ].includes(user.email.toLowerCase());

    setIsAuthorized(!!authorized);
    
    if (authorized) {
      fetchSystemStatus();
      subscribeToChanges();
    }
  };

  const fetchSystemStatus = async () => {
    try {
      // Get recent errors
      const { data: errors } = await supabase
        .from('scan_results')
        .select('*')
        .eq('resolved', false)
        .order('created_at', { ascending: false });

      const criticalErrors = errors?.filter(e => e.severity === 'high' || e.severity === 'critical') || [];
      const warnings = errors?.filter(e => e.severity === 'medium') || [];

      setErrorCount(criticalErrors.length);
      setWarningCount(warnings.length);

      if (criticalErrors.length > 0) {
        setStatus('error');
      } else if (warnings.length > 0) {
        setStatus('warning');
      } else {
        setStatus('healthy');
      }
    } catch (error) {
      console.error('Error fetching system status:', error);
    }
  };

  const subscribeToChanges = () => {
    const channel = supabase
      .channel('core_brain_widget')
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'scan_results'
        },
        () => {
          fetchSystemStatus();
        }
      )
      .subscribe();

    return () => {
      supabase.removeChannel(channel);
    };
  };

  if (!isAuthorized) return null;

  const statusConfig = {
    healthy: {
      icon: CheckCircle2,
      color: "text-green-500",
      bg: "bg-green-500/10",
      border: "border-green-500/20",
      label: "সিস্টেম সুস্থ",
      labelEn: "System Healthy"
    },
    warning: {
      icon: AlertTriangle,
      color: "text-yellow-500",
      bg: "bg-yellow-500/10",
      border: "border-yellow-500/20",
      label: "সতর্কতা",
      labelEn: "Warnings"
    },
    error: {
      icon: AlertCircle,
      color: "text-red-500",
      bg: "bg-red-500/10",
      border: "border-red-500/20",
      label: "ত্রুটি সনাক্ত",
      labelEn: "Errors Detected"
    }
  };

  const config = statusConfig[status];
  const StatusIcon = config.icon;

  return (
    <Card
      className={`p-4 cursor-pointer hover:shadow-lg transition-all ${config.bg} ${config.border} border-2`}
      onClick={() => navigate('/core-brain')}
    >
      <div className="flex items-center gap-3">
        <div className="relative">
          <div className={`absolute inset-0 ${config.color} opacity-20 blur-lg rounded-full`} />
          <Brain className={`w-6 h-6 ${config.color} relative`} />
        </div>
        <div className="flex-1">
          <div className="flex items-center gap-2">
            <h3 className="font-semibold text-sm">Core Brain</h3>
            <StatusIcon className={`w-4 h-4 ${config.color}`} />
          </div>
          <p className="text-xs text-muted-foreground">
            {config.label} • {config.labelEn}
          </p>
        </div>
        <div className="flex flex-col items-end gap-1">
          {errorCount > 0 && (
            <Badge variant="destructive" className="text-xs">
              {errorCount} ত্রুটি
            </Badge>
          )}
          {warningCount > 0 && (
            <Badge variant="secondary" className="text-xs bg-yellow-500/20 text-yellow-700">
              {warningCount} সতর্কতা
            </Badge>
          )}
          {status === 'healthy' && (
            <Badge variant="secondary" className="text-xs bg-green-500/20 text-green-700">
              ✓ সক্রিয়
            </Badge>
          )}
        </div>
      </div>
    </Card>
  );
};

export default CoreBrainWidget;
