// Full-page language selector for settings
import { Check } from "lucide-react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { ScrollArea } from "@/components/ui/scroll-area";
import { getAllLanguages, Language } from "@/lib/i18n/languages";
import { useState } from "react";
import { cn } from "@/lib/utils";

interface LanguageSelectorProps {
  selectedLanguage: string;
  onLanguageSelect: (code: string) => void;
}

export const LanguageSelector = ({ selectedLanguage, onLanguageSelect }: LanguageSelectorProps) => {
  const [searchQuery, setSearchQuery] = useState("");
  const allLanguages = getAllLanguages();
  
  const filteredLanguages = searchQuery
    ? allLanguages.filter(lang => 
        lang.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
        lang.nativeName.toLowerCase().includes(searchQuery.toLowerCase()) ||
        lang.code.toLowerCase().includes(searchQuery.toLowerCase()) ||
        lang.region.toLowerCase().includes(searchQuery.toLowerCase())
      )
    : allLanguages;
  
  // Group languages by popularity and region
  const popularLanguages = filteredLanguages.filter(lang => lang.popular);
  const otherLanguages = filteredLanguages.filter(lang => !lang.popular);

  const renderLanguageCard = (lang: Language) => (
    <Card
      key={lang.code}
      className={cn(
        "cursor-pointer transition-all hover:shadow-md",
        selectedLanguage === lang.code && "border-primary bg-primary/5"
      )}
      onClick={() => onLanguageSelect(lang.code)}
    >
      <CardContent className="flex items-center justify-between p-4">
        <div className="flex-1">
          <div className="flex items-center gap-2">
            <h3 className="font-semibold text-lg">{lang.nativeName}</h3>
            {lang.direction === 'rtl' && (
              <span className="text-xs bg-muted px-2 py-1 rounded">RTL</span>
            )}
          </div>
          <p className="text-sm text-muted-foreground">{lang.name}</p>
          <p className="text-xs text-muted-foreground mt-1">{lang.region}</p>
        </div>
        {selectedLanguage === lang.code && (
          <Check className="h-6 w-6 text-primary shrink-0" />
        )}
      </CardContent>
    </Card>
  );

  return (
    <div className="space-y-4">
      <div>
        <Input
          placeholder="Search languages by name, region, or code..."
          value={searchQuery}
          onChange={(e) => setSearchQuery(e.target.value)}
          className="max-w-md"
        />
      </div>

      {!searchQuery && popularLanguages.length > 0 && (
        <div>
          <h2 className="text-lg font-semibold mb-3">Popular Languages</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {popularLanguages.map(renderLanguageCard)}
          </div>
        </div>
      )}

      {otherLanguages.length > 0 && (
        <div>
          <h2 className="text-lg font-semibold mb-3">
            {searchQuery ? "Search Results" : "All Languages"}
          </h2>
          <ScrollArea className="h-[600px]">
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 pr-4">
              {otherLanguages.map(renderLanguageCard)}
            </div>
          </ScrollArea>
        </div>
      )}

      {filteredLanguages.length === 0 && (
        <Card>
          <CardContent className="p-8 text-center">
            <p className="text-muted-foreground">No languages found matching your search.</p>
          </CardContent>
        </Card>
      )}
    </div>
  );
};
