import { Languages, Check, ChevronDown, ChevronUp, Search } from "lucide-react";
import { Button } from "@/components/ui/button";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { ScrollArea } from "@/components/ui/scroll-area";
import { useLanguage } from "@/hooks/useLanguage";
import { getPopularLanguages, getAllLanguages } from "@/lib/i18n/languages";
import { useState } from "react";
import { Input } from "@/components/ui/input";
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from "@/components/ui/collapsible";

export const LanguageSwitcher = () => {
  const { language, setLanguage } = useLanguage();
  const [searchQuery, setSearchQuery] = useState("");
  const [showAllLanguages, setShowAllLanguages] = useState(false);
  
  const popularLanguages = getPopularLanguages();
  const allLanguages = getAllLanguages();
  
  // Filter languages - exclude popular ones from "all languages" section when not searching
  const filteredNonPopularLanguages = searchQuery
    ? allLanguages.filter(lang => 
        (lang.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
        lang.nativeName.toLowerCase().includes(searchQuery.toLowerCase()) ||
        lang.region.toLowerCase().includes(searchQuery.toLowerCase()))
      )
    : allLanguages.filter(lang => !lang.popular);

  return (
    <DropdownMenu>
      <DropdownMenuTrigger asChild>
        <Button variant="ghost" size="icon" className="relative">
          <Languages className="h-5 w-5" />
          <span className="sr-only">Change language</span>
        </Button>
      </DropdownMenuTrigger>
      <DropdownMenuContent align="end" className="w-[320px] max-h-[600px]">
        <DropdownMenuLabel className="flex items-center gap-2">
          <Search className="h-4 w-4" />
          Select Language
        </DropdownMenuLabel>
        
        {/* Search Bar */}
        <div className="px-2 pb-2">
          <Input
            placeholder="Search languages..."
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="h-8"
          />
        </div>
        <DropdownMenuSeparator />
        
        {/* Popular Languages - always visible when not searching */}
        {!searchQuery && (
          <>
            <DropdownMenuLabel className="text-xs text-muted-foreground font-semibold">
              Popular Languages
            </DropdownMenuLabel>
            <div className="max-h-[200px] overflow-y-auto">
              {popularLanguages.map((lang) => (
                <DropdownMenuItem
                  key={lang.code}
                  onClick={() => setLanguage(lang.code)}
                  className="flex items-center justify-between cursor-pointer"
                >
                  <span className="flex items-center gap-2 flex-1">
                    <span className="font-medium">{lang.nativeName}</span>
                    <span className="text-xs text-muted-foreground">({lang.name})</span>
                  </span>
                  {language === lang.code && <Check className="h-4 w-4 text-primary shrink-0" />}
                </DropdownMenuItem>
              ))}
            </div>
            <DropdownMenuSeparator />
          </>
        )}
        
        {/* All Languages - Collapsible */}
        {!searchQuery && (
          <Collapsible open={showAllLanguages} onOpenChange={setShowAllLanguages}>
            <CollapsibleTrigger asChild>
              <Button 
                variant="ghost" 
                className="w-full justify-between h-8 px-2 text-xs font-semibold text-muted-foreground hover:text-foreground"
              >
                <span>All Languages ({filteredNonPopularLanguages.length})</span>
                {showAllLanguages ? (
                  <ChevronUp className="h-4 w-4" />
                ) : (
                  <ChevronDown className="h-4 w-4" />
                )}
              </Button>
            </CollapsibleTrigger>
            <CollapsibleContent>
              <ScrollArea className="h-[250px]">
                {filteredNonPopularLanguages.map((lang) => (
                  <DropdownMenuItem
                    key={lang.code}
                    onClick={() => setLanguage(lang.code)}
                    className="flex items-center justify-between cursor-pointer"
                  >
                    <span className="flex flex-col flex-1">
                      <span className="font-medium text-sm">{lang.nativeName}</span>
                      <span className="text-xs text-muted-foreground">{lang.name} • {lang.region}</span>
                    </span>
                    {language === lang.code && <Check className="h-4 w-4 text-primary shrink-0" />}
                  </DropdownMenuItem>
                ))}
              </ScrollArea>
            </CollapsibleContent>
          </Collapsible>
        )}
        
        {/* Search Results */}
        {searchQuery && (
          <ScrollArea className="h-[300px]">
            {filteredNonPopularLanguages.length > 0 ? (
              filteredNonPopularLanguages.map((lang) => (
                <DropdownMenuItem
                  key={lang.code}
                  onClick={() => setLanguage(lang.code)}
                  className="flex items-center justify-between cursor-pointer"
                >
                  <span className="flex flex-col flex-1">
                    <span className="font-medium text-sm">{lang.nativeName}</span>
                    <span className="text-xs text-muted-foreground">{lang.name} • {lang.region}</span>
                  </span>
                  {language === lang.code && <Check className="h-4 w-4 text-primary shrink-0" />}
                </DropdownMenuItem>
              ))
            ) : (
              <div className="text-center py-8 text-sm text-muted-foreground">
                No languages found matching "{searchQuery}"
              </div>
            )}
          </ScrollArea>
        )}
      </DropdownMenuContent>
    </DropdownMenu>
  );
};
