// Phase 3: Blueprint Editor Component

import { useState } from 'react';
import { ModuleBlueprint, BlueprintEntity, BlueprintField } from '@/types/moduleBlueprint';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Plus, Trash2, Save } from 'lucide-react';

interface BlueprintEditorProps {
  blueprint: ModuleBlueprint;
  onChange: (blueprint: ModuleBlueprint) => void;
  onSave: () => void;
}

export const BlueprintEditor = ({ blueprint, onChange, onSave }: BlueprintEditorProps) => {
  const [activeTab, setActiveTab] = useState('general');

  const updateBlueprint = (updates: Partial<ModuleBlueprint>) => {
    onChange({ ...blueprint, ...updates });
  };

  const addEntity = () => {
    const newEntity: BlueprintEntity = {
      name: 'NewEntity',
      displayName: 'New Entity',
      fields: [
        { name: 'id', type: 'string', required: true, isUnique: true },
        { name: 'name', type: 'string', required: true, isSearchable: true },
      ],
      hasTimestamps: true,
      hasSoftDelete: false,
      hasOwnership: true,
    };

    updateBlueprint({
      entities: [...blueprint.entities, newEntity],
    });
  };

  const updateEntity = (index: number, updates: Partial<BlueprintEntity>) => {
    const entities = [...blueprint.entities];
    entities[index] = { ...entities[index], ...updates };
    updateBlueprint({ entities });
  };

  const deleteEntity = (index: number) => {
    updateBlueprint({
      entities: blueprint.entities.filter((_, i) => i !== index),
    });
  };

  const addField = (entityIndex: number) => {
    const entities = [...blueprint.entities];
    const newField: BlueprintField = {
      name: 'newField',
      type: 'string',
      required: false,
    };
    entities[entityIndex].fields.push(newField);
    updateBlueprint({ entities });
  };

  const updateField = (entityIndex: number, fieldIndex: number, updates: Partial<BlueprintField>) => {
    const entities = [...blueprint.entities];
    entities[entityIndex].fields[fieldIndex] = {
      ...entities[entityIndex].fields[fieldIndex],
      ...updates,
    };
    updateBlueprint({ entities });
  };

  const deleteField = (entityIndex: number, fieldIndex: number) => {
    const entities = [...blueprint.entities];
    entities[entityIndex].fields = entities[entityIndex].fields.filter((_, i) => i !== fieldIndex);
    updateBlueprint({ entities });
  };

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h2 className="text-2xl font-bold">Blueprint Editor</h2>
        <Button onClick={onSave}>
          <Save className="h-4 w-4 mr-2" />
          Save Blueprint
        </Button>
      </div>

      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList>
          <TabsTrigger value="general">General</TabsTrigger>
          <TabsTrigger value="entities">Entities</TabsTrigger>
          <TabsTrigger value="stack">Tech Stack</TabsTrigger>
        </TabsList>

        <TabsContent value="general" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>General Information</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <Label htmlFor="name">Module Name</Label>
                <Input
                  id="name"
                  value={blueprint.name}
                  onChange={(e) => updateBlueprint({ name: e.target.value })}
                />
              </div>

              <div>
                <Label htmlFor="slug">Slug</Label>
                <Input
                  id="slug"
                  value={blueprint.slug}
                  onChange={(e) => updateBlueprint({ slug: e.target.value })}
                />
              </div>

              <div>
                <Label htmlFor="description">Description</Label>
                <Input
                  id="description"
                  value={blueprint.description}
                  onChange={(e) => updateBlueprint({ description: e.target.value })}
                />
              </div>

              <div>
                <Label htmlFor="version">Version</Label>
                <Input
                  id="version"
                  value={blueprint.version}
                  onChange={(e) => updateBlueprint({ version: e.target.value })}
                />
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="entities" className="space-y-4">
          <div className="flex justify-between items-center">
            <h3 className="text-lg font-semibold">Entities</h3>
            <Button onClick={addEntity}>
              <Plus className="h-4 w-4 mr-2" />
              Add Entity
            </Button>
          </div>

          {blueprint.entities.map((entity, entityIndex) => (
            <Card key={entityIndex}>
              <CardHeader className="flex flex-row items-center justify-between">
                <CardTitle>{entity.name}</CardTitle>
                <Button
                  variant="destructive"
                  size="sm"
                  onClick={() => deleteEntity(entityIndex)}
                >
                  <Trash2 className="h-4 w-4" />
                </Button>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <Label>Name</Label>
                    <Input
                      value={entity.name}
                      onChange={(e) => updateEntity(entityIndex, { name: e.target.value })}
                    />
                  </div>
                  <div>
                    <Label>Display Name</Label>
                    <Input
                      value={entity.displayName}
                      onChange={(e) => updateEntity(entityIndex, { displayName: e.target.value })}
                    />
                  </div>
                </div>

                <div>
                  <div className="flex justify-between items-center mb-2">
                    <Label>Fields</Label>
                    <Button size="sm" onClick={() => addField(entityIndex)}>
                      <Plus className="h-3 w-3 mr-1" />
                      Add Field
                    </Button>
                  </div>

                  <div className="space-y-2">
                    {entity.fields.map((field, fieldIndex) => (
                      <div key={fieldIndex} className="flex gap-2 items-center p-2 border rounded">
                        <Input
                          placeholder="Field name"
                          value={field.name}
                          onChange={(e) => updateField(entityIndex, fieldIndex, { name: e.target.value })}
                          className="flex-1"
                        />
                        <select
                          value={field.type}
                          onChange={(e) => updateField(entityIndex, fieldIndex, { type: e.target.value as any })}
                          className="border rounded px-2 py-1"
                        >
                          <option value="string">String</option>
                          <option value="number">Number</option>
                          <option value="boolean">Boolean</option>
                          <option value="date">Date</option>
                          <option value="text">Text</option>
                          <option value="json">JSON</option>
                        </select>
                        <label className="flex items-center gap-1">
                          <input
                            type="checkbox"
                            checked={field.required}
                            onChange={(e) => updateField(entityIndex, fieldIndex, { required: e.target.checked })}
                          />
                          Required
                        </label>
                        <Button
                          variant="destructive"
                          size="sm"
                          onClick={() => deleteField(entityIndex, fieldIndex)}
                        >
                          <Trash2 className="h-3 w-3" />
                        </Button>
                      </div>
                    ))}
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </TabsContent>

        <TabsContent value="stack" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>Technology Stack</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <Label>Backend Framework</Label>
                <select
                  value={blueprint.backendStack}
                  onChange={(e) => updateBlueprint({ backendStack: e.target.value as any })}
                  className="w-full border rounded px-3 py-2"
                >
                  <option value="laravel">Laravel</option>
                  <option value="dotnet">.NET</option>
                  <option value="express">Express.js</option>
                </select>
              </div>

              <div>
                <Label>Frontend Framework</Label>
                <select
                  value={blueprint.frontendStack}
                  onChange={(e) => updateBlueprint({ frontendStack: e.target.value as any })}
                  className="w-full border rounded px-3 py-2"
                >
                  <option value="react">React</option>
                  <option value="next">Next.js</option>
                  <option value="vue">Vue.js</option>
                </select>
              </div>

              <div>
                <Label>Complexity</Label>
                <select
                  value={blueprint.complexity}
                  onChange={(e) => updateBlueprint({ complexity: e.target.value as any })}
                  className="w-full border rounded px-3 py-2"
                >
                  <option value="simple">Simple</option>
                  <option value="medium">Medium</option>
                  <option value="complex">Complex</option>
                </select>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
};
