import { Card } from "@/components/ui/card";
import { Phone, Video, PhoneIncoming, PhoneOutgoing, PhoneMissed } from "lucide-react";
import { formatCallDuration, getCallStatusText, getCallStatusColor } from "@/utils/callDuration";
import { format } from "date-fns";
import { Button } from "@/components/ui/button";

type CallHistoryItemProps = {
  call: {
    id: string;
    caller_id: string;
    callee_id: string;
    status: string;
    call_type: "voice" | "video";
    started_at: string;
    ended_at: string | null;
    created_at: string;
  };
  currentUserId: string;
  contactName: string;
  onCallBack: (contactId: string, callType: "voice" | "video") => void;
};

export const CallHistoryItem = ({
  call,
  currentUserId,
  contactName,
  onCallBack,
}: CallHistoryItemProps) => {
  const isIncoming = call.callee_id === currentUserId;
  const contactId = isIncoming ? call.caller_id : call.callee_id;
  const duration = call.ended_at ? formatCallDuration(call.started_at, call.ended_at) : "N/A";
  const statusText = getCallStatusText(call.status, call.call_type, isIncoming);
  const statusColor = getCallStatusColor(call.status);

  const getCallIcon = () => {
    if (call.status === "missed") {
      return <PhoneMissed className="h-5 w-5 text-orange-600 dark:text-orange-400" />;
    }
    if (isIncoming) {
      return <PhoneIncoming className="h-5 w-5 text-green-600 dark:text-green-400" />;
    }
    return <PhoneOutgoing className="h-5 w-5 text-blue-600 dark:text-blue-400" />;
  };

  return (
    <Card className="p-4 hover:bg-accent/50 transition-colors">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-3 flex-1">
          <div className="flex flex-col items-center gap-1">
            {getCallIcon()}
            {call.call_type === "video" ? (
              <Video className="h-3 w-3 text-muted-foreground" />
            ) : (
              <Phone className="h-3 w-3 text-muted-foreground" />
            )}
          </div>

          <div className="flex-1 min-w-0">
            <div className="flex items-center gap-2 mb-1">
              <p className="font-medium truncate">{contactName}</p>
              <span className={`text-xs ${statusColor}`}>({statusText})</span>
            </div>
            <div className="flex items-center gap-3 text-xs text-muted-foreground">
              <span>{format(new Date(call.created_at), "MMM dd, yyyy")}</span>
              <span>{format(new Date(call.created_at), "HH:mm")}</span>
              {call.status === "ended" && (
                <span className="text-primary font-medium">{duration}</span>
              )}
            </div>
          </div>
        </div>

        <div className="flex gap-2">
          <Button
            variant="ghost"
            size="icon"
            onClick={() => onCallBack(contactId, "voice")}
          >
            <Phone className="h-4 w-4" />
          </Button>
          <Button
            variant="ghost"
            size="icon"
            onClick={() => onCallBack(contactId, "video")}
          >
            <Video className="h-4 w-4" />
          </Button>
        </div>
      </div>
    </Card>
  );
};
