import { useEffect, useRef } from "react";
import { Button } from "@/components/ui/button";
import { PhoneOff, Mic, MicOff, Video, VideoOff } from "lucide-react";
import { useWebRTC } from "@/hooks/useWebRTC";
import { useState } from "react";

type VideoCallScreenProps = {
  callId: string;
  isInitiator: boolean;
  onEndCall: () => void;
  callerName?: string;
};

export const VideoCallScreen = ({
  callId,
  isInitiator,
  onEndCall,
  callerName,
}: VideoCallScreenProps) => {
  const { callState, localStream, remoteStream, endCall, toggleVideo, toggleAudio } =
    useWebRTC(callId, isInitiator);
  const localVideoRef = useRef<HTMLVideoElement>(null);
  const remoteVideoRef = useRef<HTMLVideoElement>(null);
  const [isVideoEnabled, setIsVideoEnabled] = useState(true);
  const [isAudioEnabled, setIsAudioEnabled] = useState(true);

  useEffect(() => {
    if (localVideoRef.current && localStream) {
      localVideoRef.current.srcObject = localStream;
    }
  }, [localStream]);

  useEffect(() => {
    if (remoteVideoRef.current && remoteStream) {
      remoteVideoRef.current.srcObject = remoteStream;
    }
  }, [remoteStream]);

  const handleEndCall = async () => {
    await endCall();
    onEndCall();
  };

  const handleToggleVideo = () => {
    toggleVideo();
    setIsVideoEnabled(!isVideoEnabled);
  };

  const handleToggleAudio = () => {
    toggleAudio();
    setIsAudioEnabled(!isAudioEnabled);
  };

  return (
    <div className="fixed inset-0 bg-background z-50 flex flex-col">
      <div className="flex-1 relative bg-black">
        {/* Remote video (main) */}
        <video
          ref={remoteVideoRef}
          autoPlay
          playsInline
          className="w-full h-full object-cover"
        />

        {/* Local video (picture-in-picture) */}
        <div className="absolute top-4 right-4 w-32 h-48 bg-muted rounded-lg overflow-hidden shadow-lg">
          <video
            ref={localVideoRef}
            autoPlay
            playsInline
            muted
            className="w-full h-full object-cover"
          />
        </div>

        {/* Call status */}
        <div className="absolute top-4 left-4 bg-black/50 backdrop-blur-sm px-4 py-2 rounded-full">
          <p className="text-white text-sm font-medium">
            {callState === "calling" && "Calling..."}
            {callState === "ringing" && `Incoming call from ${callerName || "User"}`}
            {callState === "connected" && "Connected"}
            {callState === "ended" && "Call Ended"}
          </p>
        </div>
      </div>

      {/* Controls */}
      <div className="p-6 bg-background/95 backdrop-blur-sm border-t">
        <div className="flex justify-center items-center gap-4 max-w-md mx-auto">
          <Button
            variant={isAudioEnabled ? "outline" : "destructive"}
            size="icon"
            className="h-14 w-14 rounded-full"
            onClick={handleToggleAudio}
          >
            {isAudioEnabled ? (
              <Mic className="h-6 w-6" />
            ) : (
              <MicOff className="h-6 w-6" />
            )}
          </Button>

          <Button
            variant="destructive"
            size="icon"
            className="h-16 w-16 rounded-full"
            onClick={handleEndCall}
          >
            <PhoneOff className="h-7 w-7" />
          </Button>

          <Button
            variant={isVideoEnabled ? "outline" : "destructive"}
            size="icon"
            className="h-14 w-14 rounded-full"
            onClick={handleToggleVideo}
          >
            {isVideoEnabled ? (
              <Video className="h-6 w-6" />
            ) : (
              <VideoOff className="h-6 w-6" />
            )}
          </Button>
        </div>
      </div>
    </div>
  );
};
