import { Camera, X } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { useCameraCapture } from "@/hooks/useCameraCapture";

interface CameraInputProps {
  onCapture: (imageText: string) => void;
}

export const CameraInput = ({ onCapture }: CameraInputProps) => {
  const { isCapturing, isProcessing, videoRef, startCamera, stopCamera, captureImage } = useCameraCapture();

  const handleCamera = async () => {
    if (!isCapturing) {
      await startCamera();
    }
  };

  const handleCapture = async () => {
    const description = await captureImage();
    if (description) {
      onCapture(description);
    }
  };

  return (
    <>
      <Button
        onClick={handleCamera}
        size="icon"
        variant="ghost"
        className="rounded-full"
        disabled={isProcessing}
      >
        <Camera className="h-5 w-5" />
      </Button>

      <Dialog open={isCapturing} onOpenChange={(open) => !open && stopCamera()}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle>Capture & Analyze Image</DialogTitle>
          </DialogHeader>
          <div className="space-y-4">
            <video
              ref={videoRef}
              className="w-full rounded-lg bg-black"
              autoPlay
              playsInline
            />
            <div className="flex gap-2">
              <Button
                onClick={handleCapture}
                disabled={isProcessing}
                className="flex-1"
              >
                {isProcessing ? "Analyzing..." : "Capture & Analyze"}
              </Button>
              <Button onClick={stopCamera} variant="outline">
                <X className="h-4 w-4" />
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    </>
  );
};