import { useState, useRef } from "react";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Send, Mic, Paperclip, Camera, Smile, Volume2, VolumeX, Phone } from "lucide-react";
import { toast } from "sonner";
import EmojiPicker from "emoji-picker-react";
import { useVoiceRecording } from "@/hooks/useVoiceRecording";

type ChatInputProps = {
  input: string;
  setInput: (value: string) => void;
  onSend: () => void;
  isLoading: boolean;
  isBangla: boolean;
  isSpeaking: boolean;
  onToggleVoice: () => void;
  onFileSelect: (file: File) => void;
};

export const ChatInput = ({
  input,
  setInput,
  onSend,
  isLoading,
  isBangla,
  isSpeaking,
  onToggleVoice,
  onFileSelect,
}: ChatInputProps) => {
  const [showEmojiPicker, setShowEmojiPicker] = useState(false);
  const { isRecording, startRecording, stopRecording } = useVoiceRecording();
  const fileInputRef = useRef<HTMLInputElement>(null);
  const cameraInputRef = useRef<HTMLInputElement>(null);

  const handleVoiceInput = async () => {
    if (isRecording) {
      const transcript = await stopRecording();
      if (transcript) {
        setInput(input + " " + transcript);
      }
    } else {
      await startRecording();
    }
  };

  const handleFileUpload = () => {
    fileInputRef.current?.click();
  };

  const handleCameraCapture = () => {
    cameraInputRef.current?.click();
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      onFileSelect(file);
      toast.success(isBangla ? "ফাইল নির্বাচিত" : "File selected");
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault();
      onSend();
    }
  };

  return (
    <div className="p-4 border-t bg-background">
      <div className="flex gap-2 max-w-4xl mx-auto">
        <Button
          type="button"
          variant="outline"
          size="icon"
          className="shrink-0"
          onClick={() => window.location.href = "/contacts"}
        >
          <Phone className="h-5 w-5" />
        </Button>

        <div className="flex-1 relative">
          <Textarea
            value={input}
            onChange={(e) => setInput(e.target.value)}
            onKeyDown={handleKeyPress}
            placeholder={isBangla ? "বার্তা লিখুন..." : "Type a message..."}
            disabled={isLoading}
            className="min-h-[48px] max-h-[120px] pr-32 rounded-3xl resize-none"
            rows={1}
          />
          
          <div className="absolute right-2 bottom-2 flex gap-1">
            <Button
              type="button"
              variant="ghost"
              size="icon"
              className="h-8 w-8"
              onClick={() => setShowEmojiPicker(!showEmojiPicker)}
            >
              <Smile className="h-4 w-4" />
            </Button>

            <Button
              type="button"
              variant="ghost"
              size="icon"
              className="h-8 w-8"
              onClick={handleFileUpload}
            >
              <Paperclip className="h-4 w-4" />
            </Button>

            <Button
              type="button"
              variant="ghost"
              size="icon"
              className="h-8 w-8"
              onClick={handleCameraCapture}
            >
              <Camera className="h-4 w-4" />
            </Button>

            <Button
              type="button"
              variant="ghost"
              size="icon"
              className={`h-8 w-8 ${isRecording ? "text-destructive" : ""}`}
              onClick={handleVoiceInput}
            >
              <Mic className="h-4 w-4" />
            </Button>
          </div>

          {showEmojiPicker && (
            <div className="absolute bottom-14 right-0 z-50">
              <EmojiPicker
                onEmojiClick={(emojiData) => {
                  setInput(input + emojiData.emoji);
                  setShowEmojiPicker(false);
                }}
              />
            </div>
          )}
        </div>

        <Button
          type="button"
          variant="outline"
          size="icon"
          className="shrink-0"
          onClick={onToggleVoice}
        >
          {isSpeaking ? <VolumeX className="h-5 w-5" /> : <Volume2 className="h-5 w-5" />}
        </Button>

        <Button
          onClick={onSend}
          size="icon"
          disabled={isLoading || !input.trim()}
          className="shrink-0"
        >
          <Send className="h-5 w-5" />
        </Button>

        <input
          ref={fileInputRef}
          type="file"
          className="hidden"
          accept="image/*,application/pdf,.doc,.docx"
          onChange={handleFileChange}
        />
        <input
          ref={cameraInputRef}
          type="file"
          className="hidden"
          accept="image/*"
          capture="environment"
          onChange={handleFileChange}
        />
      </div>
    </div>
  );
};
