import { Brain, Wifi, WifiOff, Zap, AlertCircle, Loader2 } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import {
  Tooltip,
  TooltipContent,
  TooltipProvider,
  TooltipTrigger,
} from "@/components/ui/tooltip";
import { BrainMode, BrainStatus } from "@/hooks/useHybridBrain";
import { cn } from "@/lib/utils";

interface HybridBrainIndicatorProps {
  mode: BrainMode;
  status: BrainStatus;
  isOnline: boolean;
  failureCount: number;
  onModeChange: (mode: BrainMode) => void;
}

export const HybridBrainIndicator = ({
  mode,
  status,
  isOnline,
  failureCount,
  onModeChange,
}: HybridBrainIndicatorProps) => {
  const getModeIcon = () => {
    if (status === "processing" || status === "switching") {
      return <Loader2 className="h-4 w-4 animate-spin" />;
    }
    if (status === "failed") {
      return <AlertCircle className="h-4 w-4 text-destructive" />;
    }
    
    switch (mode) {
      case "online":
        return <Wifi className="h-4 w-4 text-primary" />;
      case "offline":
        return <WifiOff className="h-4 w-4 text-muted-foreground" />;
      case "hybrid":
        return <Zap className="h-4 w-4 text-accent" />;
    }
  };

  const getModeColor = () => {
    if (status === "failed") return "destructive";
    
    switch (mode) {
      case "online":
        return "default";
      case "offline":
        return "secondary";
      case "hybrid":
        return "outline";
    }
  };

  const getModeLabel = () => {
    if (status === "processing") return "Processing...";
    if (status === "switching") return "Switching...";
    if (status === "failed") return "Failed";
    
    switch (mode) {
      case "online":
        return "Online AI";
      case "offline":
        return "Offline AI";
      case "hybrid":
        return "Hybrid AI";
    }
  };

  const getTooltipContent = () => {
    const statusText = isOnline ? "Connected" : "Offline";
    const modeText = mode === "hybrid" 
      ? "Auto-switching between online and offline"
      : mode === "online"
      ? "Using cloud AI"
      : "Using local AI";
    
    return (
      <div className="space-y-1">
        <p className="font-semibold">{getModeLabel()}</p>
        <p className="text-xs">{statusText} • {modeText}</p>
        {failureCount > 0 && (
          <p className="text-xs text-destructive">
            {failureCount} recent failures
          </p>
        )}
      </div>
    );
  };

  return (
    <div className="flex items-center gap-2">
      <TooltipProvider>
        <Tooltip>
          <TooltipTrigger asChild>
            <Badge
              variant={getModeColor()}
              className={cn(
                "flex items-center gap-2 cursor-pointer transition-all",
                status === "processing" && "animate-pulse"
              )}
            >
              {getModeIcon()}
              <span className="text-xs">{getModeLabel()}</span>
            </Badge>
          </TooltipTrigger>
          <TooltipContent>
            {getTooltipContent()}
          </TooltipContent>
        </Tooltip>
      </TooltipProvider>

      <TooltipProvider>
        <Tooltip>
          <TooltipTrigger asChild>
            <Button
              variant="ghost"
              size="icon"
              className="h-8 w-8"
              onClick={() => {
                const modes: BrainMode[] = ["hybrid", "online", "offline"];
                const currentIndex = modes.indexOf(mode);
                const nextMode = modes[(currentIndex + 1) % modes.length];
                onModeChange(nextMode);
              }}
            >
              <Brain className="h-4 w-4" />
            </Button>
          </TooltipTrigger>
          <TooltipContent>
            Click to switch AI mode
          </TooltipContent>
        </Tooltip>
      </TooltipProvider>
    </div>
  );
};