import { useKnowledgeCache } from "@/hooks/useKnowledgeCache";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Progress } from "@/components/ui/progress";
import { Badge } from "@/components/ui/badge";
import { Database, TrendingUp, Trash2, RefreshCw } from "lucide-react";
import { ScrollArea } from "@/components/ui/scroll-area";

export const KnowledgeCachePanel = () => {
  const { cacheEntries, stats, prefillCache, clearCache } = useKnowledgeCache();

  const getCategoryColor = (category: string) => {
    const colors: Record<string, string> = {
      math: "bg-blue-500",
      ict: "bg-purple-500",
      general_knowledge: "bg-green-500",
      medical: "bg-red-500",
      science: "bg-yellow-500",
      history: "bg-orange-500",
    };
    return colors[category] || "bg-gray-500";
  };

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <div>
            <CardTitle className="flex items-center gap-2">
              <Database className="h-5 w-5" />
              Knowledge Cache
            </CardTitle>
            <CardDescription>
              Offline knowledge storage for instant answers
            </CardDescription>
          </div>
          <div className="flex gap-2">
            <Button
              variant="outline"
              size="sm"
              onClick={() => prefillCache()}
            >
              <RefreshCw className="h-4 w-4 mr-2" />
              Prefill
            </Button>
            <Button
              variant="destructive"
              size="sm"
              onClick={clearCache}
            >
              <Trash2 className="h-4 w-4 mr-2" />
              Clear
            </Button>
          </div>
        </div>
      </CardHeader>
      <CardContent className="space-y-4">
        {/* Stats */}
        <div className="grid grid-cols-3 gap-4">
          <div className="space-y-1">
            <p className="text-sm text-muted-foreground">Total Entries</p>
            <p className="text-2xl font-bold">{stats.totalEntries}</p>
          </div>
          <div className="space-y-1">
            <p className="text-sm text-muted-foreground">Hit Rate</p>
            <p className="text-2xl font-bold">{stats.cacheHitRate.toFixed(1)}%</p>
          </div>
          <div className="space-y-1">
            <p className="text-sm text-muted-foreground">Top Category</p>
            <Badge className={getCategoryColor(stats.mostAccessedCategory)}>
              {stats.mostAccessedCategory}
            </Badge>
          </div>
        </div>

        {/* Hit Rate Progress */}
        <div className="space-y-2">
          <div className="flex items-center justify-between">
            <span className="text-sm font-medium">Cache Efficiency</span>
            <TrendingUp className="h-4 w-4 text-muted-foreground" />
          </div>
          <Progress value={stats.cacheHitRate} className="h-2" />
        </div>

        {/* Recent Entries */}
        <div className="space-y-2">
          <h4 className="text-sm font-semibold">Recent Entries</h4>
          <ScrollArea className="h-[200px]">
            <div className="space-y-2">
              {cacheEntries.slice(-10).reverse().map((entry, index) => (
                <div
                  key={index}
                  className="p-3 rounded-lg border bg-card space-y-1"
                >
                  <div className="flex items-center justify-between">
                    <Badge
                      variant="outline"
                      className={getCategoryColor(entry.category)}
                    >
                      {entry.category}
                    </Badge>
                    <span className="text-xs text-muted-foreground">
                      {entry.accessCount} hits
                    </span>
                  </div>
                  <p className="text-sm font-medium line-clamp-1">
                    {entry.question}
                  </p>
                  <p className="text-xs text-muted-foreground line-clamp-2">
                    {entry.answer}
                  </p>
                </div>
              ))}
            </div>
          </ScrollArea>
        </div>
      </CardContent>
    </Card>
  );
};