import { format, formatDistanceToNow } from "date-fns";
import { FunctionCallIndicator } from "./FunctionCallIndicator";

type MessageBubbleProps = {
  role: "user" | "assistant";
  content: string;
  timestamp?: Date;
  isBangla: boolean;
  showFullTime?: boolean;
  functionCall?: {
    name: string;
    isExecuting?: boolean;
  };
};

export const MessageBubble = ({ 
  role, 
  content, 
  timestamp, 
  isBangla,
  showFullTime = false,
  functionCall
}: MessageBubbleProps) => {
  const getTimeDisplay = () => {
    if (!timestamp) return "";
    
    if (showFullTime) {
      return format(timestamp, "HH:mm");
    }
    
    const now = new Date();
    const diffMinutes = Math.floor((now.getTime() - timestamp.getTime()) / 60000);
    
    if (diffMinutes < 1) {
      return isBangla ? "এখন" : "Now";
    }
    if (diffMinutes < 60) {
      return isBangla ? `${diffMinutes} মিনিট আগে` : `${diffMinutes}m ago`;
    }
    
    return format(timestamp, "HH:mm");
  };

  return (
    <div className={`flex ${role === "user" ? "justify-end" : "justify-start"} animate-fade-in group`}>
      <div
        className={`rounded-2xl px-4 py-3 max-w-[80%] shadow-sm transition-all ${
          role === "user"
            ? "bg-primary text-primary-foreground rounded-br-sm"
            : "bg-muted rounded-bl-sm"
        }`}
      >
        {functionCall && (
          <div className="mb-2">
            <FunctionCallIndicator 
              functionName={functionCall.name}
              isExecuting={functionCall.isExecuting}
            />
          </div>
        )}
        <p className="whitespace-pre-wrap text-sm leading-relaxed">{content}</p>
        {timestamp && (
          <div className="flex items-center gap-2 mt-1.5">
            <p className={`text-xs font-medium ${role === "user" ? "text-primary-foreground/80" : "text-muted-foreground"}`}>
              {getTimeDisplay()}
            </p>
            <span className={`text-[10px] opacity-0 group-hover:opacity-100 transition-opacity ${role === "user" ? "text-primary-foreground/60" : "text-muted-foreground/60"}`}>
              {format(timestamp, "HH:mm")}
            </span>
          </div>
        )}
      </div>
    </div>
  );
};
