import { useState } from "react";
import { ChevronRight, ChevronDown, Folder, File } from "lucide-react";
import { cn } from "@/lib/utils";

interface FileNode {
  name: string;
  type: "file" | "folder";
  children?: FileNode[];
}

const dummyFileTree: FileNode[] = [
  {
    name: "public",
    type: "folder",
    children: [
      { name: "index.html", type: "file" },
      { name: "favicon.ico", type: "file" },
    ],
  },
  {
    name: "src",
    type: "folder",
    children: [
      {
        name: "components",
        type: "folder",
        children: [
          { name: "Sidebar.jsx", type: "file" },
          { name: "Header.jsx", type: "file" },
          { name: "FileTree.jsx", type: "file" },
          { name: "CodeEditor.jsx", type: "file" },
          { name: "ChatUI.jsx", type: "file" },
        ],
      },
      {
        name: "layouts",
        type: "folder",
        children: [
          { name: "MainLayout.jsx", type: "file" },
        ],
      },
      {
        name: "pages",
        type: "folder",
        children: [
          { name: "Home.jsx", type: "file" },
          { name: "Preview.jsx", type: "file" },
          { name: "CodeView.jsx", type: "file" },
          { name: "Settings.jsx", type: "file" },
        ],
      },
      {
        name: "utils",
        type: "folder",
        children: [
          { name: "helpers.js", type: "file" },
        ],
      },
      { name: "App.jsx", type: "file" },
      { name: "main.jsx", type: "file" },
    ],
  },
  { name: "package.json", type: "file" },
  { name: "vite.config.js", type: "file" },
  { name: "README.md", type: "file" },
];

interface FileTreeItemProps {
  node: FileNode;
  level: number;
  onFileSelect: (path: string) => void;
  selectedFile: string | null;
  parentPath?: string;
}

const FileTreeItem = ({ node, level, onFileSelect, selectedFile, parentPath = "" }: FileTreeItemProps) => {
  const [isOpen, setIsOpen] = useState(level === 0);
  const fullPath = parentPath ? `${parentPath}/${node.name}` : node.name;

  const handleClick = () => {
    if (node.type === "folder") {
      setIsOpen(!isOpen);
    } else {
      onFileSelect(fullPath);
    }
  };

  return (
    <div>
      <button
        onClick={handleClick}
        className={cn(
          "w-full flex items-center gap-2 px-2 py-1.5 hover:bg-muted rounded transition-colors text-sm",
          selectedFile === fullPath && "bg-muted text-foreground"
        )}
        style={{ paddingLeft: `${level * 16 + 8}px` }}
      >
        {node.type === "folder" && (
          isOpen ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />
        )}
        {node.type === "folder" ? (
          <Folder className="h-4 w-4 text-muted-foreground" />
        ) : (
          <File className="h-4 w-4 text-muted-foreground" />
        )}
        <span className="text-foreground">{node.name}</span>
      </button>
      {node.type === "folder" && isOpen && node.children && (
        <div>
          {node.children.map((child, index) => (
            <FileTreeItem
              key={index}
              node={child}
              level={level + 1}
              onFileSelect={onFileSelect}
              selectedFile={selectedFile}
              parentPath={fullPath}
            />
          ))}
        </div>
      )}
    </div>
  );
};

interface BuilderFileTreeProps {
  onFileSelect: (path: string) => void;
  selectedFile: string | null;
}

export const BuilderFileTree = ({ onFileSelect, selectedFile }: BuilderFileTreeProps) => {
  return (
    <div className="w-80 border-r border-border bg-card p-4 overflow-auto">
      <h3 className="text-sm font-semibold text-foreground mb-4">Project Files</h3>
      <div className="space-y-1">
        {dummyFileTree.map((node, index) => (
          <FileTreeItem
            key={index}
            node={node}
            level={0}
            onFileSelect={onFileSelect}
            selectedFile={selectedFile}
          />
        ))}
      </div>
    </div>
  );
};
