import { useEffect, useState } from "react";
import { Card, CardContent } from "@/components/ui/card";
import { Clock, MessageSquare, FileCode, Palette, Package } from "lucide-react";
import { formatDistanceToNow } from "date-fns";
import { useNavigate } from "react-router-dom";

type RecentAction = {
  id: string;
  type: "chat" | "code" | "design" | "module";
  title: string;
  timestamp: Date;
  route: string;
};

const iconMap = {
  chat: MessageSquare,
  code: FileCode,
  design: Palette,
  module: Package,
};

export const RecentActions = () => {
  const [actions, setActions] = useState<RecentAction[]>([]);
  const navigate = useNavigate();

  useEffect(() => {
    // Load recent actions from localStorage
    const stored = localStorage.getItem("recent_actions");
    if (stored) {
      const parsed = JSON.parse(stored);
      setActions(
        parsed
          .map((a: any) => ({ ...a, timestamp: new Date(a.timestamp) }))
          .slice(0, 5)
      );
    }
  }, []);

  if (actions.length === 0) {
    return null;
  }

  return (
    <div className="space-y-4">
      <h2 className="text-lg font-semibold flex items-center gap-2">
        <Clock className="h-5 w-5" />
        Recent Actions
      </h2>
      <div className="space-y-2">
        {actions.map((action) => {
          const Icon = iconMap[action.type];
          return (
            <Card
              key={action.id}
              className="cursor-pointer hover:bg-accent/50 transition-colors"
              onClick={() => navigate(action.route)}
            >
              <CardContent className="p-3 flex items-center gap-3">
                <div className="p-2 rounded-md bg-primary/10">
                  <Icon className="h-4 w-4 text-primary" />
                </div>
                <div className="flex-1 min-w-0">
                  <p className="text-sm font-medium truncate">{action.title}</p>
                  <p className="text-xs text-muted-foreground">
                    {formatDistanceToNow(action.timestamp, { addSuffix: true })}
                  </p>
                </div>
              </CardContent>
            </Card>
          );
        })}
      </div>
    </div>
  );
};

// Helper function to add action (export for use in other components)
export const addRecentAction = (action: Omit<RecentAction, "id" | "timestamp">) => {
  const stored = localStorage.getItem("recent_actions");
  const existing: RecentAction[] = stored ? JSON.parse(stored) : [];
  
  const newAction: RecentAction = {
    ...action,
    id: Date.now().toString(),
    timestamp: new Date(),
  };
  
  const updated = [newAction, ...existing].slice(0, 10);
  localStorage.setItem("recent_actions", JSON.stringify(updated));
};
