import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Card } from "@/components/ui/card";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Badge } from "@/components/ui/badge";
import { Brain, Sparkles, Code, Zap, FileCode } from "lucide-react";
import { useAIBrain } from "@/hooks/useAIBrain";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";

interface AIBrainPanelProps {
  projectId: string;
  currentFile?: { path: string; content: string } | null;
  onCodeGenerated: (code: string) => void;
}

export const AIBrainPanel = ({ projectId, currentFile, onCodeGenerated }: AIBrainPanelProps) => {
  const [prompt, setPrompt] = useState("");
  const [action, setAction] = useState<"generate" | "refactor" | "fix" | "optimize" | "explain">("generate");
  const { generateCode, history, isGenerating } = useAIBrain(projectId);

  const handleGenerate = async () => {
    if (!prompt.trim()) return;

    const result = await generateCode.mutateAsync({
      prompt,
      context: currentFile?.content,
      filePath: currentFile?.path,
      action,
      language: "typescript",
    });

    onCodeGenerated(result.code);
    setPrompt("");
  };

  return (
    <div className="flex flex-col h-full bg-card">
      <div className="p-4 border-b border-border">
        <div className="flex items-center gap-2 mb-4">
          <Brain className="h-5 w-5 text-primary" />
          <h3 className="font-semibold text-foreground">AI Brain</h3>
          <Badge variant="secondary" className="ml-auto">
            <Sparkles className="h-3 w-3 mr-1" />
            Powered by Lovable AI
          </Badge>
        </div>

        <Select value={action} onValueChange={(v: any) => setAction(v)}>
          <SelectTrigger>
            <SelectValue />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="generate">
              <div className="flex items-center gap-2">
                <Code className="h-4 w-4" />
                Generate New Code
              </div>
            </SelectItem>
            <SelectItem value="refactor">
              <div className="flex items-center gap-2">
                <FileCode className="h-4 w-4" />
                Refactor Code
              </div>
            </SelectItem>
            <SelectItem value="fix">
              <div className="flex items-center gap-2">
                <Zap className="h-4 w-4" />
                Fix Errors
              </div>
            </SelectItem>
            <SelectItem value="optimize">
              <div className="flex items-center gap-2">
                <Sparkles className="h-4 w-4" />
                Optimize
              </div>
            </SelectItem>
            <SelectItem value="explain">
              <div className="flex items-center gap-2">
                <Brain className="h-4 w-4" />
                Explain Code
              </div>
            </SelectItem>
          </SelectContent>
        </Select>

        <Textarea
          value={prompt}
          onChange={(e) => setPrompt(e.target.value)}
          placeholder="Describe what you want to build or fix..."
          className="mt-3 min-h-24"
        />

        {currentFile && (
          <p className="text-xs text-muted-foreground mt-2">
            Context: {currentFile.path}
          </p>
        )}

        <Button
          onClick={handleGenerate}
          disabled={isGenerating || !prompt.trim()}
          className="w-full mt-3"
        >
          {isGenerating ? "Generating..." : "Generate Code"}
        </Button>
      </div>

      <ScrollArea className="flex-1 p-4">
        <div className="space-y-3">
          <h4 className="text-sm font-medium text-muted-foreground">Recent History</h4>
          {history.map((item) => (
            <Card key={item.id} className="p-3">
              <div className="flex items-start justify-between mb-2">
                <p className="text-sm font-medium text-foreground line-clamp-2">
                  {item.prompt}
                </p>
                <Badge variant={item.applied ? "default" : "secondary"}>
                  {item.applied ? "Applied" : "Pending"}
                </Badge>
              </div>
              <p className="text-xs text-muted-foreground">{item.file_path}</p>
              <p className="text-xs text-muted-foreground">
                {new Date(item.created_at).toLocaleString()}
              </p>
            </Card>
          ))}
        </div>
      </ScrollArea>
    </div>
  );
};
