import { useState } from "react";
import { Card } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Lightbulb, Code, Wand2, CheckCircle } from "lucide-react";
import { ScrollArea } from "@/components/ui/scroll-area";

interface Suggestion {
  id: string;
  type: "optimization" | "refactor" | "fix";
  title: string;
  description: string;
  code?: string;
}

interface AISuggestionsProps {
  fileName: string;
  fileContent: string;
}

export const AISuggestions = ({ fileName, fileContent }: AISuggestionsProps) => {
  const [suggestions] = useState<Suggestion[]>([
    {
      id: "1",
      type: "optimization",
      title: "Optimize React Rendering",
      description: "Use useMemo to prevent unnecessary re-renders",
    },
    {
      id: "2",
      type: "refactor",
      title: "Extract Component",
      description: "Consider extracting this logic into a separate component",
    },
    {
      id: "3",
      type: "fix",
      title: "Add Error Handling",
      description: "Add try-catch block for async operations",
    },
  ]);

  const getIcon = (type: string) => {
    switch (type) {
      case "optimization": return <Wand2 className="h-4 w-4 text-blue-500" />;
      case "refactor": return <Code className="h-4 w-4 text-purple-500" />;
      case "fix": return <CheckCircle className="h-4 w-4 text-green-500" />;
      default: return <Lightbulb className="h-4 w-4" />;
    }
  };

  return (
    <div className="flex flex-col h-full">
      <div className="p-3 border-b border-border bg-card">
        <h3 className="font-semibold text-sm text-foreground flex items-center gap-2">
          <Lightbulb className="h-4 w-4 text-yellow-500" />
          AI Suggestions
        </h3>
        <p className="text-xs text-muted-foreground mt-1">
          {fileName ? `For ${fileName}` : "Select a file to see suggestions"}
        </p>
      </div>
      <ScrollArea className="flex-1">
        <div className="p-3 space-y-2">
          {suggestions.map((suggestion) => (
            <Card key={suggestion.id} className="p-3 bg-muted/50 hover:bg-muted transition-colors">
              <div className="flex items-start gap-2">
                {getIcon(suggestion.type)}
                <div className="flex-1 min-w-0">
                  <h4 className="font-medium text-sm text-foreground">{suggestion.title}</h4>
                  <p className="text-xs text-muted-foreground mt-1">{suggestion.description}</p>
                  <Button size="sm" variant="ghost" className="mt-2 h-7 text-xs">
                    Apply
                  </Button>
                </div>
              </div>
            </Card>
          ))}
        </div>
      </ScrollArea>
    </div>
  );
};
