import { useState } from "react";
import { ChevronRight, ChevronDown, File, Folder, FolderOpen, Plus, MoreVertical, Trash, Edit, Copy, Scissors, Clipboard, Files } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuSeparator, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import { ProjectFile, useProjectFiles } from "@/hooks/useProjectFiles";
import { useFileManager } from "@/hooks/useFileManager";
import { ScrollArea } from "@/components/ui/scroll-area";

interface AdvancedFileTreeProps {
  projectId: string;
  onFileSelect: (file: ProjectFile) => void;
  selectedFile: ProjectFile | null;
}

export const AdvancedFileTree = ({ projectId, onFileSelect, selectedFile }: AdvancedFileTreeProps) => {
  const { files, isLoading } = useProjectFiles(projectId);
  const fileManager = useFileManager(projectId);
  const [expandedFolders, setExpandedFolders] = useState<Set<string>>(new Set());
  const [editingId, setEditingId] = useState<string | null>(null);
  const [newName, setNewName] = useState("");
  const [creatingIn, setCreatingIn] = useState<{ path: string; type: "file" | "folder" } | null>(null);

  const toggleFolder = (path: string) => {
    setExpandedFolders(prev => {
      const next = new Set(prev);
      if (next.has(path)) {
        next.delete(path);
      } else {
        next.add(path);
      }
      return next;
    });
  };

  const startRename = (file: ProjectFile) => {
    setEditingId(file.id);
    setNewName(file.file_name);
  };

  const finishRename = async (fileId: string) => {
    if (newName.trim()) {
      await fileManager.renameFile(fileId, newName.trim());
    }
    setEditingId(null);
    setNewName("");
  };

  const startCreating = (parentPath: string, type: "file" | "folder") => {
    setCreatingIn({ path: parentPath, type });
    setNewName("");
  };

  const finishCreating = async () => {
    if (!creatingIn || !newName.trim()) return;

    if (creatingIn.type === "folder") {
      await fileManager.createFolder(creatingIn.path, newName.trim());
    } else {
      const filePath = creatingIn.path ? `${creatingIn.path}/${newName.trim()}` : newName.trim();
      await fileManager.createFile({
        fileName: newName.trim(),
        filePath,
        fileType: "file",
        content: "",
        parentId: null,
      });
    }

    setCreatingIn(null);
    setNewName("");
  };

  const buildTree = (files: ProjectFile[]) => {
    const tree: any = {};
    
    files.forEach(file => {
      const parts = file.file_path.split("/");
      let current = tree;
      
      parts.forEach((part, i) => {
        if (!current[part]) {
          current[part] = {
            file: i === parts.length - 1 ? file : null,
            children: {},
          };
        }
        current = current[part].children;
      });
    });
    
    return tree;
  };

  const renderTree = (tree: any, path: string = "", level: number = 0): JSX.Element[] => {
    return Object.keys(tree).map(key => {
      const node = tree[key];
      const fullPath = path ? `${path}/${key}` : key;
      const isFolder = !node.file || node.file.file_type === "folder";
      const isExpanded = expandedFolders.has(fullPath);
      const isSelected = selectedFile?.id === node.file?.id;
      const isEditing = editingId === node.file?.id;

      return (
        <div key={fullPath}>
          <div
            className={`flex items-center gap-1 px-2 py-1 hover:bg-accent cursor-pointer group ${
              isSelected ? "bg-accent" : ""
            }`}
            style={{ paddingLeft: `${level * 12 + 8}px` }}
          >
            {isFolder && (
              <button onClick={() => toggleFolder(fullPath)} className="p-0.5">
                {isExpanded ? (
                  <ChevronDown className="h-4 w-4" />
                ) : (
                  <ChevronRight className="h-4 w-4" />
                )}
              </button>
            )}
            
            {!isFolder && <div className="w-5" />}

            {isFolder ? (
              isExpanded ? (
                <FolderOpen className="h-4 w-4 text-primary" />
              ) : (
                <Folder className="h-4 w-4 text-primary" />
              )
            ) : (
              <File className="h-4 w-4 text-muted-foreground" />
            )}

            {isEditing ? (
              <Input
                value={newName}
                onChange={(e) => setNewName(e.target.value)}
                onBlur={() => finishRename(node.file.id)}
                onKeyDown={(e) => {
                  if (e.key === "Enter") finishRename(node.file.id);
                  if (e.key === "Escape") setEditingId(null);
                }}
                className="h-6 text-sm"
                autoFocus
              />
            ) : (
              <span
                className="flex-1 text-sm truncate"
                onClick={() => !isFolder && node.file && onFileSelect(node.file)}
              >
                {key}
              </span>
            )}

            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button
                  size="sm"
                  variant="ghost"
                  className="h-6 w-6 p-0 opacity-0 group-hover:opacity-100"
                >
                  <MoreVertical className="h-4 w-4" />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="end">
                {isFolder && (
                  <>
                    <DropdownMenuItem onClick={() => startCreating(fullPath, "file")}>
                      <Plus className="h-4 w-4 mr-2" />
                      New File
                    </DropdownMenuItem>
                    <DropdownMenuItem onClick={() => startCreating(fullPath, "folder")}>
                      <Plus className="h-4 w-4 mr-2" />
                      New Folder
                    </DropdownMenuItem>
                    <DropdownMenuSeparator />
                  </>
                )}
                <DropdownMenuItem onClick={() => node.file && startRename(node.file)}>
                  <Edit className="h-4 w-4 mr-2" />
                  Rename
                </DropdownMenuItem>
                <DropdownMenuItem onClick={() => node.file && fileManager.copyFile(node.file)}>
                  <Copy className="h-4 w-4 mr-2" />
                  Copy
                </DropdownMenuItem>
                <DropdownMenuItem onClick={() => node.file && fileManager.cutFile(node.file)}>
                  <Scissors className="h-4 w-4 mr-2" />
                  Cut
                </DropdownMenuItem>
                {fileManager.clipboard && (
                  <DropdownMenuItem onClick={() => fileManager.pasteFile(fullPath)}>
                    <Clipboard className="h-4 w-4 mr-2" />
                    Paste
                  </DropdownMenuItem>
                )}
                <DropdownMenuItem onClick={() => node.file && fileManager.duplicateFile(node.file.id)}>
                  <Files className="h-4 w-4 mr-2" />
                  Duplicate
                </DropdownMenuItem>
                <DropdownMenuSeparator />
                <DropdownMenuItem
                  onClick={() => node.file && fileManager.deleteFile(node.file.id)}
                  className="text-destructive"
                >
                  <Trash className="h-4 w-4 mr-2" />
                  Delete
                </DropdownMenuItem>
              </DropdownMenuContent>
            </DropdownMenu>
          </div>

          {isFolder && isExpanded && (
            <>
              {creatingIn?.path === fullPath && (
                <div
                  className="flex items-center gap-1 px-2 py-1"
                  style={{ paddingLeft: `${(level + 1) * 12 + 8}px` }}
                >
                  {creatingIn.type === "folder" ? (
                    <Folder className="h-4 w-4 text-primary" />
                  ) : (
                    <File className="h-4 w-4 text-muted-foreground" />
                  )}
                  <Input
                    value={newName}
                    onChange={(e) => setNewName(e.target.value)}
                    onBlur={finishCreating}
                    onKeyDown={(e) => {
                      if (e.key === "Enter") finishCreating();
                      if (e.key === "Escape") setCreatingIn(null);
                    }}
                    placeholder={`New ${creatingIn.type}...`}
                    className="h-6 text-sm"
                    autoFocus
                  />
                </div>
              )}
              {renderTree(node.children, fullPath, level + 1)}
            </>
          )}
        </div>
      );
    });
  };

  if (isLoading) {
    return <div className="p-4 text-muted-foreground">Loading files...</div>;
  }

  const tree = buildTree(files);

  return (
    <div className="flex flex-col h-full bg-card">
      <div className="p-3 border-b border-border flex items-center justify-between">
        <h3 className="font-semibold text-sm text-foreground">Files</h3>
        <div className="flex items-center gap-1">
          <Button
            size="sm"
            variant="ghost"
            onClick={() => startCreating("", "file")}
            className="h-7 w-7 p-0"
          >
            <Plus className="h-4 w-4" />
          </Button>
          <Button
            size="sm"
            variant="ghost"
            onClick={() => startCreating("", "folder")}
            className="h-7 w-7 p-0"
          >
            <Folder className="h-4 w-4" />
          </Button>
        </div>
      </div>

      <ScrollArea className="flex-1">
        {creatingIn?.path === "" && (
          <div className="flex items-center gap-1 px-2 py-1">
            {creatingIn.type === "folder" ? (
              <Folder className="h-4 w-4 text-primary" />
            ) : (
              <File className="h-4 w-4 text-muted-foreground" />
            )}
            <Input
              value={newName}
              onChange={(e) => setNewName(e.target.value)}
              onBlur={finishCreating}
              onKeyDown={(e) => {
                if (e.key === "Enter") finishCreating();
                if (e.key === "Escape") setCreatingIn(null);
              }}
              placeholder={`New ${creatingIn.type}...`}
              className="h-6 text-sm"
              autoFocus
            />
          </div>
        )}
        {renderTree(tree)}
      </ScrollArea>
    </div>
  );
};
