import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Package, Download, CheckCircle, AlertCircle, Loader2 } from "lucide-react";
import { useProductionExport } from "@/hooks/useProductionExport";
import { Progress } from "@/components/ui/progress";
import { Alert, AlertDescription } from "@/components/ui/alert";

interface BuildExportPanelProps {
  projectId: string;
}

export const BuildExportPanel = ({ projectId }: BuildExportPanelProps) => {
  const { exportProduction, exporting, progress, status } = useProductionExport(projectId);
  const [buildInfo, setBuildInfo] = useState<{
    timestamp?: string;
    size?: string;
    files?: number;
  }>({});

  const handleExport = async () => {
    const info = await exportProduction("smart-friend-core-production");
    if (info) {
      setBuildInfo(info);
    }
  };

  return (
    <div className="flex flex-col h-full overflow-hidden">
      <div className="p-4 border-b border-border">
        <h2 className="text-lg font-semibold text-foreground flex items-center gap-2">
          <Package className="h-5 w-5 text-primary" />
          Production Build & Export
        </h2>
        <p className="text-sm text-muted-foreground mt-1">
          Export complete production-ready ZIP for cPanel deployment
        </p>
      </div>

      <div className="flex-1 overflow-y-auto p-4 space-y-4">
        {/* Export Status */}
        {exporting && (
          <Card>
            <CardHeader>
              <CardTitle className="text-base flex items-center gap-2">
                <Loader2 className="h-4 w-4 animate-spin" />
                Building Production Package...
              </CardTitle>
            </CardHeader>
            <CardContent>
              <Progress value={progress} className="mb-2" />
              <p className="text-sm text-muted-foreground">{status}</p>
            </CardContent>
          </Card>
        )}

        {/* Export Button */}
        <Card>
          <CardHeader>
            <CardTitle className="text-base">Export Production ZIP</CardTitle>
            <CardDescription>
              Download complete production-ready package for manual cPanel deployment
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Button
              onClick={handleExport}
              disabled={exporting}
              className="w-full gap-2"
              size="lg"
            >
              {exporting ? (
                <>
                  <Loader2 className="h-4 w-4 animate-spin" />
                  Preparing Export...
                </>
              ) : (
                <>
                  <Download className="h-4 w-4" />
                  Download Production ZIP
                </>
              )}
            </Button>

            {buildInfo.timestamp && (
              <div className="mt-4 p-3 bg-muted rounded-lg space-y-2">
                <div className="flex items-center gap-2 text-sm text-foreground">
                  <CheckCircle className="h-4 w-4 text-green-500" />
                  <span className="font-medium">Last Export: {buildInfo.timestamp}</span>
                </div>
                <div className="text-xs text-muted-foreground space-y-1">
                  <p>Size: {buildInfo.size}</p>
                  <p>Files: {buildInfo.files}</p>
                </div>
              </div>
            )}
          </CardContent>
        </Card>

        {/* What's Included */}
        <Card>
          <CardHeader>
            <CardTitle className="text-base">Package Contents</CardTitle>
            <CardDescription>Production ZIP includes:</CardDescription>
          </CardHeader>
          <CardContent className="space-y-2">
            <div className="flex items-start gap-2 text-sm">
              <CheckCircle className="h-4 w-4 text-green-500 mt-0.5 flex-shrink-0" />
              <span>All source files (src/, public/)</span>
            </div>
            <div className="flex items-start gap-2 text-sm">
              <CheckCircle className="h-4 w-4 text-green-500 mt-0.5 flex-shrink-0" />
              <span>Complete package.json with all dependencies</span>
            </div>
            <div className="flex items-start gap-2 text-sm">
              <CheckCircle className="h-4 w-4 text-green-500 mt-0.5 flex-shrink-0" />
              <span>Build configuration (vite.config.ts, tsconfig.json)</span>
            </div>
            <div className="flex items-start gap-2 text-sm">
              <CheckCircle className="h-4 w-4 text-green-500 mt-0.5 flex-shrink-0" />
              <span>Environment setup (.env.example)</span>
            </div>
            <div className="flex items-start gap-2 text-sm">
              <CheckCircle className="h-4 w-4 text-green-500 mt-0.5 flex-shrink-0" />
              <span>Deployment guide (README-DEPLOY.md)</span>
            </div>
            <div className="flex items-start gap-2 text-sm">
              <CheckCircle className="h-4 w-4 text-green-500 mt-0.5 flex-shrink-0" />
              <span>Production optimizations enabled</span>
            </div>
          </CardContent>
        </Card>

        {/* Deployment Instructions */}
        <Card>
          <CardHeader>
            <CardTitle className="text-base">cPanel Deployment Steps</CardTitle>
          </CardHeader>
          <CardContent className="space-y-3 text-sm">
            <div>
              <p className="font-medium text-foreground mb-1">1. Extract ZIP</p>
              <p className="text-muted-foreground">Upload and extract the ZIP file to your cPanel public_html directory</p>
            </div>
            <div>
              <p className="font-medium text-foreground mb-1">2. Install Dependencies</p>
              <p className="text-muted-foreground">SSH into your server and run: npm install</p>
            </div>
            <div>
              <p className="font-medium text-foreground mb-1">3. Configure Environment</p>
              <p className="text-muted-foreground">Copy .env.example to .env and add your Supabase credentials</p>
            </div>
            <div>
              <p className="font-medium text-foreground mb-1">4. Build for Production</p>
              <p className="text-muted-foreground">Run: npm run build</p>
            </div>
            <div>
              <p className="font-medium text-foreground mb-1">5. Deploy</p>
              <p className="text-muted-foreground">Point your domain to the dist/ folder or copy dist contents to public_html</p>
            </div>
          </CardContent>
        </Card>

        {/* Warnings */}
        <Alert>
          <AlertCircle className="h-4 w-4" />
          <AlertDescription className="text-sm">
            <strong>Important:</strong> Make sure to update environment variables in .env with your production Supabase credentials before building.
          </AlertDescription>
        </Alert>
      </div>
    </div>
  );
};
