import { useState } from "react";
import { DiffEditor } from "@monaco-editor/react";
import { Button } from "@/components/ui/button";
import { Check, X, RotateCcw } from "lucide-react";
import { useToast } from "@/hooks/use-toast";

interface CodeDiffViewerProps {
  original: string;
  modified: string;
  fileName: string;
  onApply: (code: string) => void;
  onReject: () => void;
}

export const CodeDiffViewer = ({
  original,
  modified,
  fileName,
  onApply,
  onReject,
}: CodeDiffViewerProps) => {
  const { toast } = useToast();
  const [showUndo, setShowUndo] = useState(false);

  const handleApply = () => {
    onApply(modified);
    setShowUndo(true);
    toast({ title: "Changes applied successfully" });
  };

  const handleUndo = () => {
    onApply(original);
    setShowUndo(false);
    toast({ title: "Changes reverted" });
  };

  const getLanguage = (fileName: string) => {
    const ext = fileName.split(".").pop()?.toLowerCase();
    const map: Record<string, string> = {
      js: "javascript", jsx: "javascript", ts: "typescript", tsx: "typescript",
      php: "php", py: "python", dart: "dart", java: "java",
      html: "html", css: "css", json: "json", md: "markdown",
    };
    return map[ext || ""] || "plaintext";
  };

  return (
    <div className="flex flex-col h-full">
      <div className="flex items-center justify-between p-3 border-b border-border bg-card">
        <div>
          <h3 className="font-semibold text-foreground">Code Diff: {fileName}</h3>
          <p className="text-xs text-muted-foreground">Review changes before applying</p>
        </div>
        <div className="flex gap-2">
          {showUndo && (
            <Button size="sm" variant="outline" onClick={handleUndo}>
              <RotateCcw className="h-4 w-4 mr-1" />
              Undo
            </Button>
          )}
          <Button size="sm" variant="destructive" onClick={onReject}>
            <X className="h-4 w-4 mr-1" />
            Reject
          </Button>
          <Button size="sm" onClick={handleApply} disabled={showUndo}>
            <Check className="h-4 w-4 mr-1" />
            Apply
          </Button>
        </div>
      </div>
      <div className="flex-1">
        <DiffEditor
          original={original}
          modified={modified}
          language={getLanguage(fileName)}
          theme="vs-dark"
          options={{
            readOnly: true,
            renderSideBySide: true,
            minimap: { enabled: false },
          }}
        />
      </div>
    </div>
  );
};
