import { useState, useEffect } from "react";
import { X } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { ProjectFile, useProjectFiles } from "@/hooks/useProjectFiles";
import { cn } from "@/lib/utils";

interface CodeEditorProps {
  projectId: string;
  selectedFile: ProjectFile | null;
  openFiles: ProjectFile[];
  activeTab: string | null;
  onTabChange: (fileId: string) => void;
  onCloseTab: (fileId: string) => void;
  showPreview: boolean;
}

export const CodeEditor = ({
  projectId,
  selectedFile,
  openFiles,
  activeTab,
  onTabChange,
  onCloseTab,
  showPreview,
}: CodeEditorProps) => {
  const { updateFile } = useProjectFiles(projectId);
  const [code, setCode] = useState("");
  const [hasChanges, setHasChanges] = useState(false);

  useEffect(() => {
    if (selectedFile?.content) {
      setCode(selectedFile.content);
      setHasChanges(false);
    }
  }, [selectedFile?.id]);

  const handleSave = async () => {
    if (selectedFile && hasChanges) {
      await updateFile.mutateAsync({ id: selectedFile.id, content: code });
      setHasChanges(false);
    }
  };

  if (!selectedFile) {
    return (
      <div className="flex-1 flex items-center justify-center bg-background">
        <p className="text-muted-foreground">Select a file to start editing</p>
      </div>
    );
  }

  return (
    <div className="flex-1 flex flex-col">
      <div className="flex items-center gap-1 border-b border-border bg-card px-2">
        {openFiles.map((file) => (
          <div
            key={file.id}
            className={cn(
              "flex items-center gap-2 px-3 py-2 cursor-pointer border-b-2 transition-colors",
              activeTab === file.id
                ? "border-primary bg-muted/50"
                : "border-transparent hover:bg-muted/30"
            )}
            onClick={() => onTabChange(file.id)}
          >
            <span className="text-sm text-foreground">{file.file_name}</span>
            <Button
              size="sm"
              variant="ghost"
              onClick={(e) => {
                e.stopPropagation();
                onCloseTab(file.id);
              }}
              className="h-4 w-4 p-0 hover:bg-destructive/20"
            >
              <X className="h-3 w-3" />
            </Button>
          </div>
        ))}
      </div>

      <div className={cn("flex-1 flex", showPreview ? "gap-2" : "")}>
        <div className={cn("flex flex-col", showPreview ? "w-1/2" : "flex-1")}>
          <div className="flex items-center justify-between p-2 border-b border-border bg-card">
            <span className="text-sm font-medium text-foreground">{selectedFile.file_name}</span>
            <Button
              size="sm"
              onClick={handleSave}
              disabled={!hasChanges || updateFile.isPending}
              className="h-7"
            >
              {updateFile.isPending ? "Saving..." : "Save"}
            </Button>
          </div>

          <Textarea
            value={code}
            onChange={(e) => {
              setCode(e.target.value);
              setHasChanges(true);
            }}
            className="flex-1 font-mono text-sm border-none rounded-none resize-none"
            placeholder="Start typing..."
          />
        </div>

        {showPreview && (
          <div className="w-1/2 border-l border-border bg-card p-4">
            <h3 className="text-sm font-semibold text-foreground mb-2">Preview</h3>
            <div className="bg-background rounded-lg p-4 border border-border">
              <p className="text-muted-foreground text-sm">Live preview will appear here</p>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};
