import { useState } from "react";
import { FileInfo } from "@/core/devBrain/DevBrainFileOps";
import { useDevBrain } from "@/hooks/useDevBrain";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Brain, Sparkles, AlertCircle, Upload, Loader2 } from "lucide-react";
import { CodeDiffViewer } from "./CodeDiffViewer";

interface DevBrainPanelProps {
  selectedFile: FileInfo | null;
}

export const DevBrainPanel = ({ selectedFile }: DevBrainPanelProps) => {
  const [refactorGoal, setRefactorGoal] = useState("");
  const [errorLog, setErrorLog] = useState("");
  const {
    analyzeFile,
    analyzeError,
    suggestRefactor,
    generateUpgradePlan,
    applyChanges,
    isAnalyzing,
    analysis,
    diff,
    newCode,
    clearAnalysis,
  } = useDevBrain();

  const handleAnalyzeFile = async () => {
    if (!selectedFile || !selectedFile.content) return;
    await analyzeFile(selectedFile.path, selectedFile.content);
  };

  const handleAnalyzeError = async () => {
    if (!errorLog.trim()) return;
    await analyzeError(errorLog);
  };

  const handleSuggestRefactor = async () => {
    if (!selectedFile || !selectedFile.content || !refactorGoal.trim()) return;
    await suggestRefactor({
      filePath: selectedFile.path,
      selectedCode: selectedFile.content,
      goal: refactorGoal,
    });
  };

  const handleApplyChanges = () => {
    if (!selectedFile || !newCode || !diff) return;
    applyChanges(selectedFile.path, newCode, diff);
  };

  if (diff && newCode) {
    return (
      <div className="h-full">
        <CodeDiffViewer
          original={selectedFile?.content || ""}
          modified={newCode}
          fileName={selectedFile?.name || "file"}
          onApply={handleApplyChanges}
          onReject={clearAnalysis}
        />
      </div>
    );
  }

  return (
    <div className="h-full overflow-auto p-6 space-y-6">
      <div className="flex items-center gap-3">
        <Brain className="h-6 w-6 text-primary" />
        <div>
          <h2 className="text-2xl font-bold text-foreground">Dev Brain</h2>
          <p className="text-sm text-muted-foreground">
            AI-powered code analysis and improvements
          </p>
        </div>
      </div>

      {/* File Analysis */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Sparkles className="h-5 w-5" />
            Analyze File
          </CardTitle>
          <CardDescription>
            Get AI suggestions for the selected file
          </CardDescription>
        </CardHeader>
        <CardContent>
          {selectedFile ? (
            <div className="space-y-4">
              <div className="p-3 bg-muted rounded-md">
                <p className="text-sm font-mono">{selectedFile.path}</p>
                <p className="text-xs text-muted-foreground mt-1">
                  {selectedFile.language} • Ready for analysis
                </p>
              </div>
              <Button
                onClick={handleAnalyzeFile}
                disabled={isAnalyzing}
                className="w-full"
              >
                {isAnalyzing ? (
                  <>
                    <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                    Analyzing...
                  </>
                ) : (
                  <>
                    <Brain className="h-4 w-4 mr-2" />
                    Analyze Code
                  </>
                )}
              </Button>
            </div>
          ) : (
            <div className="text-center p-6 text-muted-foreground">
              <Upload className="h-8 w-8 mx-auto mb-2" />
              <p>Select a file from the inspector</p>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Refactor Suggestion */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Sparkles className="h-5 w-5" />
            Suggest Refactor
          </CardTitle>
          <CardDescription>
            Tell the AI what you want to improve
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <Textarea
            placeholder="e.g., Improve performance, add error handling, make it more modular..."
            value={refactorGoal}
            onChange={(e) => setRefactorGoal(e.target.value)}
            rows={3}
            disabled={!selectedFile}
          />
          <Button
            onClick={handleSuggestRefactor}
            disabled={!selectedFile || !refactorGoal.trim() || isAnalyzing}
            className="w-full"
          >
            {isAnalyzing ? (
              <>
                <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                Generating...
              </>
            ) : (
              <>
                <Sparkles className="h-4 w-4 mr-2" />
                Generate Refactor
              </>
            )}
          </Button>
        </CardContent>
      </Card>

      {/* Error Analysis */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <AlertCircle className="h-5 w-5" />
            Analyze Error
          </CardTitle>
          <CardDescription>Paste an error log for analysis</CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <Textarea
            placeholder="Paste error logs here..."
            value={errorLog}
            onChange={(e) => setErrorLog(e.target.value)}
            rows={4}
          />
          <Button
            onClick={handleAnalyzeError}
            disabled={!errorLog.trim() || isAnalyzing}
            className="w-full"
          >
            {isAnalyzing ? (
              <>
                <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                Analyzing...
              </>
            ) : (
              <>
                <AlertCircle className="h-4 w-4 mr-2" />
                Analyze Error
              </>
            )}
          </Button>
        </CardContent>
      </Card>

      {/* Upgrade Plan */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Upload className="h-5 w-5" />
            Generate Upgrade Plan
          </CardTitle>
          <CardDescription>
            Get a comprehensive project upgrade plan
          </CardDescription>
        </CardHeader>
        <CardContent>
          <Button
            onClick={generateUpgradePlan}
            disabled={isAnalyzing}
            className="w-full"
            variant="outline"
          >
            {isAnalyzing ? (
              <>
                <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                Generating...
              </>
            ) : (
              <>
                <Upload className="h-4 w-4 mr-2" />
                Generate Plan
              </>
            )}
          </Button>
        </CardContent>
      </Card>

      {/* Analysis Results */}
      {analysis && (
        <Card className="border-primary">
          <CardHeader>
            <CardTitle>{analysis.summary}</CardTitle>
            <CardDescription className="capitalize">{analysis.type.replace("_", " ")}</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="prose prose-sm max-w-none dark:prose-invert">
              <p className="whitespace-pre-wrap">{analysis.details}</p>
            </div>
            {analysis.affectedFiles && analysis.affectedFiles.length > 0 && (
              <div className="mt-4">
                <p className="text-sm font-medium mb-2">Affected Files:</p>
                <ul className="text-sm space-y-1">
                  {analysis.affectedFiles.map((file) => (
                    <li key={file} className="font-mono text-xs">{file}</li>
                  ))}
                </ul>
              </div>
            )}
          </CardContent>
        </Card>
      )}
    </div>
  );
};
