import { useEffect } from "react";
import { useDevBrain } from "@/hooks/useDevBrain";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { History, FileCode, CheckCircle, Clock, XCircle } from "lucide-react";
import { formatDistanceToNow } from "date-fns";

export const DevHistoryViewer = () => {
  const { logs, loadLogs } = useDevBrain();

  useEffect(() => {
    loadLogs(50);
  }, []);

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "applied":
        return <CheckCircle className="h-4 w-4 text-green-500" />;
      case "pending":
        return <Clock className="h-4 w-4 text-yellow-500" />;
      case "failed":
        return <XCircle className="h-4 w-4 text-red-500" />;
      default:
        return <Clock className="h-4 w-4" />;
    }
  };

  const getStatusBadgeVariant = (status: string) => {
    switch (status) {
      case "applied":
        return "default";
      case "pending":
        return "secondary";
      case "failed":
        return "destructive";
      default:
        return "outline";
    }
  };

  return (
    <div className="h-full flex flex-col">
      <div className="p-4 border-b border-border">
        <div className="flex items-center gap-2">
          <History className="h-5 w-5 text-primary" />
          <div>
            <h3 className="font-semibold text-foreground">Action History</h3>
            <p className="text-xs text-muted-foreground">
              {logs.length} actions logged
            </p>
          </div>
        </div>
      </div>

      <ScrollArea className="flex-1">
        <div className="p-4 space-y-3">
          {logs.length === 0 ? (
            <Card>
              <CardContent className="pt-6 text-center text-muted-foreground">
                <History className="h-12 w-12 mx-auto mb-3 opacity-50" />
                <p>No actions yet</p>
                <p className="text-xs mt-1">
                  Actions will appear here as you use Dev Brain
                </p>
              </CardContent>
            </Card>
          ) : (
            logs.map((log) => (
              <Card key={log.id} className="hover:border-primary transition-colors">
                <CardHeader className="pb-3">
                  <div className="flex items-start justify-between gap-2">
                    <div className="flex items-center gap-2 flex-1">
                      {getStatusIcon(log.status)}
                      <CardTitle className="text-sm">{log.description}</CardTitle>
                    </div>
                    <Badge variant={getStatusBadgeVariant(log.status)}>
                      {log.status}
                    </Badge>
                  </div>
                  <CardDescription className="flex items-center gap-2">
                    <span className="capitalize">{log.action_type.replace("_", " ")}</span>
                    <span>•</span>
                    <span>
                      {formatDistanceToNow(new Date(log.created_at), {
                        addSuffix: true,
                      })}
                    </span>
                  </CardDescription>
                </CardHeader>
                {log.file_path && (
                  <CardContent className="pt-0">
                    <div className="flex items-center gap-2 text-xs">
                      <FileCode className="h-3 w-3 text-muted-foreground" />
                      <code className="text-muted-foreground">{log.file_path}</code>
                    </div>
                    {log.applied_at && (
                      <p className="text-xs text-green-600 mt-2">
                        Applied {formatDistanceToNow(new Date(log.applied_at), {
                          addSuffix: true,
                        })}
                      </p>
                    )}
                  </CardContent>
                )}
              </Card>
            ))
          )}
        </div>
      </ScrollArea>
    </div>
  );
};
