import { useState } from "react";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import {
  Bug,
  Wrench,
  History,
  Database,
  Download,
  RefreshCw,
  CheckCircle,
  AlertCircle,
  Clock,
} from "lucide-react";
import { useErrorScanner } from "@/hooks/useErrorScanner";
import { useLogAnalyzer } from "@/hooks/useLogAnalyzer";
import { useAutoFix } from "@/hooks/useAutoFix";
import { useBackupSystem } from "@/hooks/useBackupSystem";
import { useLaravelGenerator } from "@/hooks/useLaravelGenerator";

export const DevOpsPanel = () => {
  const [activeTab, setActiveTab] = useState("scanner");
  const { scanProject, errors, isScanning } = useErrorScanner();
  const { analyzeLogs, analysis, analyzing } = useLogAnalyzer();
  const { generateFix, applyPatch, patches, generating } = useAutoFix();
  const { createSnapshot, rollback, snapshots, creating } = useBackupSystem();
  const { generateAPI, exportBackend, projects, generating: genLaravel } = useLaravelGenerator();

  return (
    <div className="h-full flex flex-col bg-card">
      <div className="border-b border-border p-4">
        <h2 className="text-lg font-semibold text-foreground">DevOps Dashboard</h2>
        <p className="text-sm text-muted-foreground">Auto DevOps + Backend Builder</p>
      </div>

      <Tabs value={activeTab} onValueChange={setActiveTab} className="flex-1 flex flex-col">
        <TabsList className="w-full justify-start rounded-none border-b">
          <TabsTrigger value="scanner" className="gap-2">
            <Bug className="h-4 w-4" />
            Scanner
          </TabsTrigger>
          <TabsTrigger value="fixes" className="gap-2">
            <Wrench className="h-4 w-4" />
            Fixes
          </TabsTrigger>
          <TabsTrigger value="backups" className="gap-2">
            <History className="h-4 w-4" />
            Backups
          </TabsTrigger>
          <TabsTrigger value="backend" className="gap-2">
            <Database className="h-4 w-4" />
            Backend
          </TabsTrigger>
        </TabsList>

        <ScrollArea className="flex-1">
          <TabsContent value="scanner" className="p-4 space-y-4">
            <Button
              onClick={scanProject}
              disabled={isScanning}
              className="w-full"
            >
              {isScanning ? (
                <>
                  <RefreshCw className="mr-2 h-4 w-4 animate-spin" />
                  Scanning...
                </>
              ) : (
                <>
                  <Bug className="mr-2 h-4 w-4" />
                  Scan Project
                </>
              )}
            </Button>

            {errors.length > 0 && (
              <div className="space-y-2">
                <h3 className="font-medium text-foreground">Detected Issues ({errors.length})</h3>
                {errors.map((error, idx) => (
                  <Card key={idx} className="p-3">
                    <div className="flex items-start justify-between gap-2">
                      <div className="flex-1">
                        <div className="flex items-center gap-2 mb-1">
                          <Badge variant={error.severity === "critical" ? "destructive" : "secondary"}>
                            {error.severity}
                          </Badge>
                          <Badge variant="outline">{error.type}</Badge>
                        </div>
                        <p className="text-sm text-foreground">{error.message}</p>
                        <p className="text-xs text-muted-foreground mt-1">
                          {error.file}:{error.line}
                        </p>
                      </div>
                      <Button
                        size="sm"
                        onClick={() => generateFix(error)}
                        disabled={generating}
                      >
                        Fix
                      </Button>
                    </div>
                  </Card>
                ))}
              </div>
            )}

            {analysis && (
              <Card className="p-4 space-y-2">
                <h3 className="font-medium text-foreground">Root Cause Analysis</h3>
                <p className="text-sm text-foreground">{analysis.rootCause}</p>
                <div className="bg-muted p-2 rounded text-xs">
                  <strong>Suggested Fix:</strong> {analysis.suggestedFix}
                </div>
                <div className="text-xs text-muted-foreground">
                  Affected: {analysis.affectedFiles.join(", ")}
                </div>
              </Card>
            )}
          </TabsContent>

          <TabsContent value="fixes" className="p-4 space-y-4">
            <div className="flex items-center justify-between">
              <h3 className="font-medium text-foreground">Generated Fixes</h3>
              <Badge variant="secondary">{patches.length}</Badge>
            </div>

            {patches.length === 0 ? (
              <p className="text-sm text-muted-foreground text-center py-8">
                No fixes generated yet. Scan for errors first.
              </p>
            ) : (
              <div className="space-y-2">
                {patches.map((patch) => (
                  <Card key={patch.id} className="p-3">
                    <div className="flex items-start justify-between gap-2">
                      <div className="flex-1">
                        <div className="flex items-center gap-2 mb-1">
                          {patch.applied ? (
                            <CheckCircle className="h-4 w-4 text-green-500" />
                          ) : (
                            <AlertCircle className="h-4 w-4 text-yellow-500" />
                          )}
                          <span className="text-sm font-medium text-foreground">
                            {patch.description}
                          </span>
                        </div>
                        <p className="text-xs text-muted-foreground">{patch.file}</p>
                      </div>
                      {!patch.applied && (
                        <Button
                          size="sm"
                          onClick={() => applyPatch(patch.id)}
                        >
                          Apply
                        </Button>
                      )}
                    </div>
                  </Card>
                ))}
              </div>
            )}
          </TabsContent>

          <TabsContent value="backups" className="p-4 space-y-4">
            <Button
              onClick={() => createSnapshot("Manual backup")}
              disabled={creating}
              className="w-full"
            >
              {creating ? (
                <>
                  <RefreshCw className="mr-2 h-4 w-4 animate-spin" />
                  Creating...
                </>
              ) : (
                <>
                  <History className="mr-2 h-4 w-4" />
                  Create Snapshot
                </>
              )}
            </Button>

            <div className="space-y-2">
              <h3 className="font-medium text-foreground">Snapshots ({snapshots.length}/10)</h3>
              {snapshots.map((snapshot) => (
                <Card key={snapshot.id} className="p-3">
                  <div className="flex items-start justify-between gap-2">
                    <div className="flex-1">
                      <p className="text-sm font-medium text-foreground">{snapshot.description}</p>
                      <div className="flex items-center gap-2 mt-1 text-xs text-muted-foreground">
                        <Clock className="h-3 w-3" />
                        {new Date(snapshot.timestamp).toLocaleString()}
                      </div>
                      <p className="text-xs text-muted-foreground mt-1">
                        {snapshot.filesCount} files • {snapshot.size}
                      </p>
                    </div>
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => rollback(snapshot.id)}
                    >
                      Restore
                    </Button>
                  </div>
                </Card>
              ))}
            </div>
          </TabsContent>

          <TabsContent value="backend" className="p-4 space-y-4">
            <Button
              onClick={() => generateAPI({
                name: "MyAPI",
                resource: "Post",
                fields: [
                  { name: "title", type: "string" },
                  { name: "content", type: "text" },
                  { name: "published", type: "boolean" },
                ]
              })}
              disabled={genLaravel}
              className="w-full"
            >
              {genLaravel ? (
                <>
                  <RefreshCw className="mr-2 h-4 w-4 animate-spin" />
                  Generating...
                </>
              ) : (
                <>
                  <Database className="mr-2 h-4 w-4" />
                  Generate Laravel API
                </>
              )}
            </Button>

            <div className="space-y-2">
              <h3 className="font-medium text-foreground">Laravel Projects ({projects.length})</h3>
              {projects.map((project) => (
                <Card key={project.name} className="p-3">
                  <div className="flex items-start justify-between gap-2">
                    <div className="flex-1">
                      <p className="text-sm font-medium text-foreground">{project.name}</p>
                      <p className="text-xs text-muted-foreground mt-1">
                        {project.components.length} files generated
                      </p>
                      <div className="flex gap-1 mt-2">
                        {Array.from(new Set(project.components.map(c => c.type))).map(type => (
                          <Badge key={type} variant="outline" className="text-xs">
                            {type}
                          </Badge>
                        ))}
                      </div>
                    </div>
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => exportBackend(project.name)}
                    >
                      <Download className="mr-2 h-4 w-4" />
                      Export
                    </Button>
                  </div>
                </Card>
              ))}
            </div>
          </TabsContent>
        </ScrollArea>
      </Tabs>
    </div>
  );
};
