import { useState } from "react";
import { ChevronRight, ChevronDown, File, Folder, Plus, FolderPlus } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { useProjectFiles, ProjectFile } from "@/hooks/useProjectFiles";
import { cn } from "@/lib/utils";

interface FileExplorerProps {
  projectId: string;
  onFileSelect: (file: ProjectFile) => void;
}

export const FileExplorer = ({ projectId, onFileSelect }: FileExplorerProps) => {
  const { files, createFile } = useProjectFiles(projectId);
  const [expandedFolders, setExpandedFolders] = useState<Set<string>>(new Set(["/"]));
  const [creatingFile, setCreatingFile] = useState<string | null>(null);
  const [newFileName, setNewFileName] = useState("");

  const toggleFolder = (path: string) => {
    const newExpanded = new Set(expandedFolders);
    if (newExpanded.has(path)) {
      newExpanded.delete(path);
    } else {
      newExpanded.add(path);
    }
    setExpandedFolders(newExpanded);
  };

  const handleCreateFile = async (parentPath: string, isFolder: boolean) => {
    if (!newFileName) return;
    
    const filePath = parentPath === "/" ? `/${newFileName}` : `${parentPath}/${newFileName}`;
    await createFile.mutateAsync({
      fileName: newFileName,
      filePath,
      fileType: isFolder ? "folder" : "file",
      content: isFolder ? undefined : "",
    });
    
    setCreatingFile(null);
    setNewFileName("");
  };

  const buildTree = () => {
    const rootFiles = files.filter((f) => !f.file_path.includes("/") || f.file_path.split("/").length === 2);
    return rootFiles;
  };

  const FileTreeItem = ({ file, level = 0 }: { file: ProjectFile; level?: number }) => {
    const isExpanded = expandedFolders.has(file.file_path);
    const children = files.filter((f) => 
      f.file_path.startsWith(file.file_path + "/") &&
      f.file_path.split("/").length === file.file_path.split("/").length + 1
    );

    return (
      <div>
        <div
          className={cn(
            "flex items-center gap-2 px-2 py-1.5 cursor-pointer hover:bg-muted/50 rounded-md",
            "text-sm"
          )}
          style={{ paddingLeft: `${level * 12 + 8}px` }}
          onClick={() => {
            if (file.file_type === "folder") {
              toggleFolder(file.file_path);
            } else {
              onFileSelect(file);
            }
          }}
        >
          {file.file_type === "folder" ? (
            <>
              {isExpanded ? (
                <ChevronDown className="h-4 w-4 text-muted-foreground" />
              ) : (
                <ChevronRight className="h-4 w-4 text-muted-foreground" />
              )}
              <Folder className="h-4 w-4 text-primary" />
            </>
          ) : (
            <>
              <div className="w-4" />
              <File className="h-4 w-4 text-muted-foreground" />
            </>
          )}
          <span className="text-foreground">{file.file_name}</span>
        </div>

        {file.file_type === "folder" && isExpanded && (
          <div>
            {children.map((child) => (
              <FileTreeItem key={child.id} file={child} level={level + 1} />
            ))}
            {creatingFile === file.file_path && (
              <div style={{ paddingLeft: `${(level + 1) * 12 + 8}px` }} className="px-2 py-1">
                <Input
                  value={newFileName}
                  onChange={(e) => setNewFileName(e.target.value)}
                  onKeyDown={(e) => {
                    if (e.key === "Enter") handleCreateFile(file.file_path, false);
                    if (e.key === "Escape") setCreatingFile(null);
                  }}
                  placeholder="filename.tsx"
                  className="h-7 text-sm"
                  autoFocus
                />
              </div>
            )}
          </div>
        )}
      </div>
    );
  };

  return (
    <div className="h-full flex flex-col">
      <div className="p-3 border-b border-border flex items-center justify-between">
        <h3 className="text-sm font-semibold text-foreground">Files</h3>
        <div className="flex gap-1">
          <Button
            size="sm"
            variant="ghost"
            onClick={() => setCreatingFile("/")}
            className="h-7 w-7 p-0"
          >
            <Plus className="h-4 w-4" />
          </Button>
          <Button
            size="sm"
            variant="ghost"
            onClick={() => {
              setNewFileName("");
              setCreatingFile("folder");
            }}
            className="h-7 w-7 p-0"
          >
            <FolderPlus className="h-4 w-4" />
          </Button>
        </div>
      </div>

      <div className="flex-1 overflow-auto p-2">
        {buildTree().map((file) => (
          <FileTreeItem key={file.id} file={file} />
        ))}
        
        {creatingFile === "/" && (
          <div className="px-2 py-1">
            <Input
              value={newFileName}
              onChange={(e) => setNewFileName(e.target.value)}
              onKeyDown={(e) => {
                if (e.key === "Enter") handleCreateFile("/", false);
                if (e.key === "Escape") setCreatingFile(null);
              }}
              placeholder="filename.tsx"
              className="h-7 text-sm"
              autoFocus
            />
          </div>
        )}
      </div>
    </div>
  );
};
