import { Info, FileText, Calendar, User } from "lucide-react";
import { Card } from "@/components/ui/card";
import { ProjectFile } from "@/hooks/useProjectFiles";

interface FileInfoProps {
  file: ProjectFile | null;
}

export const FileInfo = ({ file }: FileInfoProps) => {
  if (!file) {
    return (
      <div className="flex flex-col items-center justify-center h-full p-4 text-center">
        <Info className="h-12 w-12 text-muted-foreground mb-3" />
        <p className="text-sm text-muted-foreground">Select a file to view details</p>
      </div>
    );
  }

  const fileSize = file.content ? new Blob([file.content]).size : 0;
  const lines = file.content ? file.content.split("\n").length : 0;

  return (
    <div className="p-4 space-y-4">
      <div>
        <h3 className="font-semibold text-sm text-foreground flex items-center gap-2 mb-3">
          <Info className="h-4 w-4" />
          File Information
        </h3>
        <Card className="p-3 bg-muted/50 space-y-3">
          <div className="flex items-start gap-2">
            <FileText className="h-4 w-4 text-muted-foreground mt-0.5" />
            <div className="flex-1 min-w-0">
              <p className="text-xs text-muted-foreground">File Name</p>
              <p className="text-sm font-medium text-foreground truncate">{file.file_name}</p>
            </div>
          </div>
          
          <div className="flex items-start gap-2">
            <FileText className="h-4 w-4 text-muted-foreground mt-0.5" />
            <div className="flex-1 min-w-0">
              <p className="text-xs text-muted-foreground">Path</p>
              <p className="text-sm font-medium text-foreground truncate">{file.file_path}</p>
            </div>
          </div>

          <div className="flex items-start gap-2">
            <FileText className="h-4 w-4 text-muted-foreground mt-0.5" />
            <div className="flex-1">
              <p className="text-xs text-muted-foreground">Type</p>
              <p className="text-sm font-medium text-foreground">{file.file_type}</p>
            </div>
          </div>

          <div className="flex items-start gap-2">
            <FileText className="h-4 w-4 text-muted-foreground mt-0.5" />
            <div className="flex-1">
              <p className="text-xs text-muted-foreground">Size</p>
              <p className="text-sm font-medium text-foreground">{(fileSize / 1024).toFixed(2)} KB</p>
            </div>
          </div>

          <div className="flex items-start gap-2">
            <FileText className="h-4 w-4 text-muted-foreground mt-0.5" />
            <div className="flex-1">
              <p className="text-xs text-muted-foreground">Lines</p>
              <p className="text-sm font-medium text-foreground">{lines}</p>
            </div>
          </div>

          <div className="flex items-start gap-2">
            <Calendar className="h-4 w-4 text-muted-foreground mt-0.5" />
            <div className="flex-1 min-w-0">
              <p className="text-xs text-muted-foreground">Created</p>
              <p className="text-sm font-medium text-foreground">
                {new Date(file.created_at || "").toLocaleDateString()}
              </p>
            </div>
          </div>

          <div className="flex items-start gap-2">
            <Calendar className="h-4 w-4 text-muted-foreground mt-0.5" />
            <div className="flex-1 min-w-0">
              <p className="text-xs text-muted-foreground">Modified</p>
              <p className="text-sm font-medium text-foreground">
                {new Date(file.updated_at || "").toLocaleDateString()}
              </p>
            </div>
          </div>
        </Card>
      </div>
    </div>
  );
};
