import { useState, useEffect } from "react";
import { DevBrainFileOps, FileInfo } from "@/core/devBrain/DevBrainFileOps";
import { File, Folder, ChevronRight, ChevronDown } from "lucide-react";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Button } from "@/components/ui/button";

interface FileInspectorProps {
  onFileSelect: (file: FileInfo) => void;
  selectedFile: FileInfo | null;
}

export const FileInspector = ({ onFileSelect, selectedFile }: FileInspectorProps) => {
  const [files, setFiles] = useState<FileInfo[]>([]);
  const [loading, setLoading] = useState(true);
  const [expandedFolders, setExpandedFolders] = useState<Set<string>>(new Set());
  const fileOps = new DevBrainFileOps();

  useEffect(() => {
    loadFiles();
  }, []);

  const loadFiles = async () => {
    setLoading(true);
    try {
      const projectFiles = await fileOps.getProjectFiles();
      setFiles(projectFiles);
    } catch (error) {
      console.error("Error loading files:", error);
    } finally {
      setLoading(false);
    }
  };

  const toggleFolder = (folderPath: string) => {
    const newExpanded = new Set(expandedFolders);
    if (newExpanded.has(folderPath)) {
      newExpanded.delete(folderPath);
    } else {
      newExpanded.add(folderPath);
    }
    setExpandedFolders(newExpanded);
  };

  const renderFileTree = (items: FileInfo[], level: number = 0) => {
    return items.map((item) => {
      const isExpanded = expandedFolders.has(item.path);
      const isSelected = selectedFile?.path === item.path;

      return (
        <div key={item.path} style={{ paddingLeft: `${level * 16}px` }}>
          <Button
            variant="ghost"
            size="sm"
            className={`w-full justify-start gap-2 ${
              isSelected ? "bg-primary/10" : ""
            }`}
            onClick={() => {
              if (item.type === "folder") {
                toggleFolder(item.path);
              } else {
                onFileSelect(item);
              }
            }}
          >
            {item.type === "folder" ? (
              <>
                {isExpanded ? (
                  <ChevronDown className="h-4 w-4" />
                ) : (
                  <ChevronRight className="h-4 w-4" />
                )}
                <Folder className="h-4 w-4" />
              </>
            ) : (
              <File className="h-4 w-4" />
            )}
            <span className="text-sm truncate">{item.name}</span>
          </Button>
        </div>
      );
    });
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-full">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary" />
      </div>
    );
  }

  return (
    <div className="h-full flex flex-col">
      <div className="p-4 border-b border-border">
        <h3 className="font-semibold text-foreground">Project Files</h3>
        <p className="text-xs text-muted-foreground mt-1">
          Select a file to analyze
        </p>
      </div>
      <ScrollArea className="flex-1">
        <div className="p-2">
          {files.length === 0 ? (
            <div className="text-center p-8 text-muted-foreground">
              <p>No files found</p>
              <p className="text-xs mt-2">Upload files to get started</p>
            </div>
          ) : (
            renderFileTree(files)
          )}
        </div>
      </ScrollArea>
    </div>
  );
};
