import { useState } from "react";
import { FileExplorer } from "./FileExplorer";
import { MonacoCodeEditor } from "./MonacoCodeEditor";
import { TopBar } from "./TopBar";
import { BottomPanel } from "./BottomPanel";
import { LivePreview } from "./LivePreview";
import { AIBrainPanel } from "./AIBrainPanel";
import { DevOpsPanel } from "./DevOpsPanel";
import { SelfUpgradePanel } from "./SelfUpgradePanel";
import { BuildExportPanel } from "./BuildExportPanel";
import { ProjectFile } from "@/hooks/useProjectFiles";
import { Button } from "@/components/ui/button";
import { Brain, Eye, Wrench, Sparkles, Package } from "lucide-react";

interface IDELayoutProps {
  projectId: string;
}

export const IDELayout = ({ projectId }: IDELayoutProps) => {
  const [selectedFile, setSelectedFile] = useState<ProjectFile | null>(null);
  const [openFiles, setOpenFiles] = useState<ProjectFile[]>([]);
  const [activeTab, setActiveTab] = useState<string | null>(null);
  const [rightPanel, setRightPanel] = useState<"preview" | "ai" | "devops" | "upgrade" | "build">("preview");
  const [showPreview, setShowPreview] = useState(false);

  const handleFileSelect = (file: ProjectFile) => {
    setSelectedFile(file);
    if (file.file_type === "file" && !openFiles.find((f) => f.id === file.id)) {
      setOpenFiles([...openFiles, file]);
    }
    setActiveTab(file.id);
  };

  const handleCloseTab = (fileId: string) => {
    setOpenFiles(openFiles.filter((f) => f.id !== fileId));
    if (activeTab === fileId) {
      const remaining = openFiles.filter((f) => f.id !== fileId);
      setActiveTab(remaining.length > 0 ? remaining[0].id : null);
      setSelectedFile(remaining.length > 0 ? remaining[0] : null);
    }
  };

  return (
    <div className="h-screen flex flex-col bg-background">
      <TopBar 
        projectId={projectId}
        showPreview={showPreview}
        onTogglePreview={() => setShowPreview(!showPreview)}
      />
      
      <div className="flex-1 flex overflow-hidden">
        <div className="w-64 border-r border-border bg-card">
          <FileExplorer projectId={projectId} onFileSelect={handleFileSelect} />
        </div>

        <div className="flex-1 flex flex-col">
          <MonacoCodeEditor
            projectId={projectId}
            selectedFile={selectedFile}
            openFiles={openFiles}
            activeTab={activeTab}
            onTabChange={setActiveTab}
            onCloseTab={handleCloseTab}
          />
          
          <BottomPanel projectId={projectId} />
        </div>

        <div className="w-96 border-l border-border bg-card flex flex-col">
          <div className="border-b border-border p-2 flex gap-1">
            <Button
              variant={rightPanel === "preview" ? "secondary" : "ghost"}
              size="sm"
              onClick={() => setRightPanel("preview")}
              className="gap-2"
            >
              <Eye className="h-4 w-4" />
              Preview
            </Button>
            <Button
              variant={rightPanel === "ai" ? "secondary" : "ghost"}
              size="sm"
              onClick={() => setRightPanel("ai")}
              className="gap-2"
            >
              <Brain className="h-4 w-4" />
              AI Brain
            </Button>
            <Button
              variant={rightPanel === "devops" ? "secondary" : "ghost"}
              size="sm"
              onClick={() => setRightPanel("devops")}
              className="gap-2"
            >
              <Wrench className="h-4 w-4" />
              DevOps
            </Button>
            <Button
              variant={rightPanel === "upgrade" ? "secondary" : "ghost"}
              size="sm"
              onClick={() => setRightPanel("upgrade")}
              className="gap-2"
            >
              <Sparkles className="h-4 w-4" />
              Upgrade
            </Button>
            <Button
              variant={rightPanel === "build" ? "secondary" : "ghost"}
              size="sm"
              onClick={() => setRightPanel("build")}
              className="gap-2"
            >
              <Package className="h-4 w-4" />
              Build
            </Button>
          </div>

          <div className="flex-1 overflow-hidden">
            {rightPanel === "preview" && <LivePreview projectId={projectId} />}
            {rightPanel === "ai" && <AIBrainPanel projectId={projectId} onCodeGenerated={(code) => console.log("Code generated:", code)} />}
            {rightPanel === "devops" && <DevOpsPanel />}
            {rightPanel === "upgrade" && <SelfUpgradePanel />}
            {rightPanel === "build" && <BuildExportPanel projectId={projectId} />}
          </div>
        </div>
      </div>
    </div>
  );
};
