import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { useMemory } from "@/hooks/useMemory";
import { Brain, Save, Trash2 } from "lucide-react";

interface MemorySettingsProps {
  projectId: string;
}

export const MemorySettings = ({ projectId }: MemorySettingsProps) => {
  const { memories, saveMemory } = useMemory(projectId);
  const [newKey, setNewKey] = useState("");
  const [newValue, setNewValue] = useState("");
  const [memoryType, setMemoryType] = useState("preference");

  const handleSave = async () => {
    if (!newKey || !newValue) return;

    await saveMemory.mutateAsync({
      type: memoryType,
      key: newKey,
      value: JSON.parse(newValue),
    });

    setNewKey("");
    setNewValue("");
  };

  const projectMemories = memories.filter((m) => m.project_id === projectId);
  const globalMemories = memories.filter((m) => !m.project_id);

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center gap-2">
        <Brain className="h-6 w-6 text-primary" />
        <div>
          <h2 className="text-2xl font-bold text-foreground">Memory System</h2>
          <p className="text-muted-foreground">LangCache-style long-term memory</p>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Add New Memory</CardTitle>
          <CardDescription>Store preferences, settings, or instructions</CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-2 gap-4">
            <div className="space-y-2">
              <label className="text-sm font-medium text-foreground">Type</label>
              <select
                value={memoryType}
                onChange={(e) => setMemoryType(e.target.value)}
                className="w-full p-2 border border-border rounded-md bg-background text-foreground"
              >
                <option value="preference">Preference</option>
                <option value="instruction">Instruction</option>
                <option value="architecture">Architecture</option>
                <option value="setting">Setting</option>
              </select>
            </div>
            <div className="space-y-2">
              <label className="text-sm font-medium text-foreground">Key</label>
              <Input
                value={newKey}
                onChange={(e) => setNewKey(e.target.value)}
                placeholder="e.g., theme, framework"
              />
            </div>
          </div>

          <div className="space-y-2">
            <label className="text-sm font-medium text-foreground">Value (JSON)</label>
            <Textarea
              value={newValue}
              onChange={(e) => setNewValue(e.target.value)}
              placeholder='{"value": "dark", "reason": "user preference"}'
              className="font-mono text-sm"
            />
          </div>

          <Button onClick={handleSave} disabled={saveMemory.isPending} className="gap-2">
            <Save className="h-4 w-4" />
            Save Memory
          </Button>
        </CardContent>
      </Card>

      <div className="space-y-4">
        <h3 className="text-lg font-semibold text-foreground">Project Memories</h3>
        {projectMemories.length === 0 ? (
          <p className="text-muted-foreground text-sm">No project memories yet</p>
        ) : (
          <div className="space-y-2">
            {projectMemories.map((memory) => (
              <Card key={memory.id}>
                <CardContent className="p-4">
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <div className="flex items-center gap-2 mb-1">
                        <span className="text-xs px-2 py-1 bg-primary/10 text-primary rounded">
                          {memory.memory_type}
                        </span>
                        <span className="text-sm font-medium text-foreground">{memory.memory_key}</span>
                      </div>
                      <pre className="text-xs text-muted-foreground mt-2 p-2 bg-muted rounded">
                        {JSON.stringify(memory.memory_value, null, 2)}
                      </pre>
                    </div>
                    <Button size="sm" variant="ghost" className="text-destructive">
                      <Trash2 className="h-4 w-4" />
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        )}
      </div>

      <div className="space-y-4">
        <h3 className="text-lg font-semibold text-foreground">Global Memories</h3>
        {globalMemories.length === 0 ? (
          <p className="text-muted-foreground text-sm">No global memories yet</p>
        ) : (
          <div className="space-y-2">
            {globalMemories.map((memory) => (
              <Card key={memory.id}>
                <CardContent className="p-4">
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <div className="flex items-center gap-2 mb-1">
                        <span className="text-xs px-2 py-1 bg-secondary/10 text-secondary rounded">
                          {memory.memory_type}
                        </span>
                        <span className="text-sm font-medium text-foreground">{memory.memory_key}</span>
                      </div>
                      <pre className="text-xs text-muted-foreground mt-2 p-2 bg-muted rounded">
                        {JSON.stringify(memory.memory_value, null, 2)}
                      </pre>
                    </div>
                    <Button size="sm" variant="ghost" className="text-destructive">
                      <Trash2 className="h-4 w-4" />
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        )}
      </div>
    </div>
  );
};
