import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { useModuleGenerator } from "@/hooks/useModuleGenerator";
import { Loader2, Package } from "lucide-react";

interface ModuleGeneratorProps {
  projectId: string;
}

export const ModuleGenerator = ({ projectId }: ModuleGeneratorProps) => {
  const { templates, isLoading, generateModule } = useModuleGenerator();
  const [selectedTemplate, setSelectedTemplate] = useState<string | null>(null);
  const [customName, setCustomName] = useState("");

  const handleGenerate = async () => {
    if (!selectedTemplate) return;
    
    await generateModule.mutateAsync({
      templateSlug: selectedTemplate,
      projectId,
      customName: customName || undefined,
    });

    setSelectedTemplate(null);
    setCustomName("");
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-64">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </div>
    );
  }

  return (
    <div className="p-6 space-y-6">
      <div>
        <h2 className="text-2xl font-bold text-foreground mb-2">Module Generator</h2>
        <p className="text-muted-foreground">
          Generate complete modules with one click. All Smart Seba modules available.
        </p>
      </div>

      {selectedTemplate && (
        <Card>
          <CardHeader>
            <CardTitle>Customize Module</CardTitle>
            <CardDescription>Enter a custom name (optional)</CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <Input
              value={customName}
              onChange={(e) => setCustomName(e.target.value)}
              placeholder="Custom module name"
            />
            <div className="flex gap-2">
              <Button onClick={handleGenerate} disabled={generateModule.isPending}>
                {generateModule.isPending && <Loader2 className="h-4 w-4 animate-spin mr-2" />}
                Generate Module
              </Button>
              <Button variant="outline" onClick={() => setSelectedTemplate(null)}>
                Cancel
              </Button>
            </div>
          </CardContent>
        </Card>
      )}

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
        {templates.map((template) => (
          <Card
            key={template.id}
            className="cursor-pointer hover:border-primary transition-colors"
            onClick={() => setSelectedTemplate(template.slug)}
          >
            <CardHeader>
              <div className="flex items-center gap-2">
                <Package className="h-5 w-5 text-primary" />
                <CardTitle className="text-lg">{template.name}</CardTitle>
              </div>
              <CardDescription>{template.description}</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="flex items-center justify-between">
                <span className="text-xs text-muted-foreground">{template.category}</span>
                {template.is_multi_tenant && (
                  <span className="text-xs px-2 py-1 bg-primary/10 text-primary rounded">
                    Multi-tenant
                  </span>
                )}
              </div>
            </CardContent>
          </Card>
        ))}
      </div>
    </div>
  );
};
