import { useEffect, useRef, useState } from "react";
import Editor, { OnMount } from "@monaco-editor/react";
import { Button } from "@/components/ui/button";
import { X, Save, Copy, Download } from "lucide-react";
import { ProjectFile, useProjectFiles } from "@/hooks/useProjectFiles";
import { cn } from "@/lib/utils";
import { useToast } from "@/hooks/use-toast";

interface MonacoCodeEditorProps {
  projectId: string;
  selectedFile: ProjectFile | null;
  openFiles: ProjectFile[];
  activeTab: string | null;
  onTabChange: (fileId: string) => void;
  onCloseTab: (fileId: string) => void;
}

export const MonacoCodeEditor = ({
  projectId,
  selectedFile,
  openFiles,
  activeTab,
  onTabChange,
  onCloseTab,
}: MonacoCodeEditorProps) => {
  const { updateFile } = useProjectFiles(projectId);
  const { toast } = useToast();
  const [code, setCode] = useState("");
  const [hasChanges, setHasChanges] = useState(false);
  const editorRef = useRef<any>(null);

  useEffect(() => {
    if (selectedFile?.content) {
      setCode(selectedFile.content);
      setHasChanges(false);
    }
  }, [selectedFile?.id]);

  const handleEditorDidMount: OnMount = (editor) => {
    editorRef.current = editor;
  };

  const handleSave = async () => {
    if (selectedFile && hasChanges) {
      await updateFile.mutateAsync({ id: selectedFile.id, content: code });
      setHasChanges(false);
      toast({ title: "File saved successfully" });
    }
  };

  const handleCopy = () => {
    navigator.clipboard.writeText(code);
    toast({ title: "Code copied to clipboard" });
  };

  const handleDownload = () => {
    if (!selectedFile) return;
    const blob = new Blob([code], { type: "text/plain" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = selectedFile.file_name;
    a.click();
    URL.revokeObjectURL(url);
    toast({ title: "File downloaded" });
  };

  const getLanguage = (fileName: string) => {
    const ext = fileName.split(".").pop()?.toLowerCase();
    const languageMap: Record<string, string> = {
      js: "javascript",
      jsx: "javascript",
      ts: "typescript",
      tsx: "typescript",
      php: "php",
      py: "python",
      dart: "dart",
      java: "java",
      html: "html",
      css: "css",
      json: "json",
      md: "markdown",
      sql: "sql",
      xml: "xml",
      yaml: "yaml",
      yml: "yaml",
    };
    return languageMap[ext || ""] || "plaintext";
  };

  if (!selectedFile) {
    return (
      <div className="flex-1 flex items-center justify-center bg-background">
        <div className="text-center">
          <p className="text-muted-foreground mb-2">No file selected</p>
          <p className="text-sm text-muted-foreground">Select a file from the explorer to start editing</p>
        </div>
      </div>
    );
  }

  return (
    <div className="flex-1 flex flex-col h-full">
      {/* Tabs */}
      <div className="flex items-center gap-1 border-b border-border bg-card px-2 overflow-x-auto">
        {openFiles.map((file) => (
          <div
            key={file.id}
            className={cn(
              "flex items-center gap-2 px-3 py-2 cursor-pointer border-b-2 transition-colors whitespace-nowrap",
              activeTab === file.id
                ? "border-primary bg-muted/50"
                : "border-transparent hover:bg-muted/30"
            )}
            onClick={() => onTabChange(file.id)}
          >
            <span className="text-sm text-foreground">{file.file_name}</span>
            <Button
              size="sm"
              variant="ghost"
              onClick={(e) => {
                e.stopPropagation();
                onCloseTab(file.id);
              }}
              className="h-4 w-4 p-0 hover:bg-destructive/20"
            >
              <X className="h-3 w-3" />
            </Button>
          </div>
        ))}
      </div>

      {/* Editor Header */}
      <div className="flex items-center justify-between p-2 border-b border-border bg-card">
        <div className="flex items-center gap-2">
          <span className="text-sm font-medium text-foreground">{selectedFile.file_name}</span>
          {hasChanges && <span className="text-xs text-warning">● Unsaved</span>}
        </div>
        <div className="flex items-center gap-2">
          <Button size="sm" variant="ghost" onClick={handleCopy} className="h-7">
            <Copy className="h-4 w-4 mr-1" />
            Copy
          </Button>
          <Button size="sm" variant="ghost" onClick={handleDownload} className="h-7">
            <Download className="h-4 w-4 mr-1" />
            Download
          </Button>
          <Button
            size="sm"
            onClick={handleSave}
            disabled={!hasChanges || updateFile.isPending}
            className="h-7"
          >
            <Save className="h-4 w-4 mr-1" />
            {updateFile.isPending ? "Saving..." : "Save"}
          </Button>
        </div>
      </div>

      {/* Monaco Editor */}
      <div className="flex-1 overflow-hidden">
        <Editor
          height="100%"
          defaultLanguage={getLanguage(selectedFile.file_name)}
          language={getLanguage(selectedFile.file_name)}
          value={code}
          onChange={(value) => {
            setCode(value || "");
            setHasChanges(true);
          }}
          onMount={handleEditorDidMount}
          theme="vs-dark"
          options={{
            minimap: { enabled: true },
            fontSize: 14,
            lineNumbers: "on",
            rulers: [],
            wordWrap: "on",
            scrollBeyondLastLine: false,
            automaticLayout: true,
            tabSize: 2,
            formatOnPaste: true,
            formatOnType: true,
          }}
        />
      </div>

      {/* Status Bar */}
      <div className="flex items-center justify-between px-4 py-1 border-t border-border bg-card text-xs text-muted-foreground">
        <span>{getLanguage(selectedFile.file_name).toUpperCase()}</span>
        <span>UTF-8</span>
      </div>
    </div>
  );
};
