import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Badge } from "@/components/ui/badge";
import { Play, Square, Terminal, Clock } from "lucide-react";
import { useRuntimeEngine } from "@/hooks/useRuntimeEngine";

interface RuntimePreviewProps {
  code: string;
  language: string;
}

export const RuntimePreview = ({ code, language }: RuntimePreviewProps) => {
  const { executeCode, clearResult, isRunning, result } = useRuntimeEngine();

  return (
    <div className="flex flex-col h-full bg-card">
      <div className="p-4 border-b border-border">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-2">
            <Terminal className="h-5 w-5 text-primary" />
            <h3 className="font-semibold text-foreground">Runtime Preview</h3>
          </div>
          <div className="flex items-center gap-2">
            {result && (
              <Badge variant="secondary">
                <Clock className="h-3 w-3 mr-1" />
                {result.executionTime.toFixed(2)}ms
              </Badge>
            )}
            <Button
              size="sm"
              onClick={() => executeCode(code, language)}
              disabled={isRunning}
            >
              {isRunning ? (
                <>
                  <Square className="h-4 w-4 mr-2" />
                  Running
                </>
              ) : (
                <>
                  <Play className="h-4 w-4 mr-2" />
                  Run
                </>
              )}
            </Button>
            {result && (
              <Button size="sm" variant="outline" onClick={clearResult}>
                Clear
              </Button>
            )}
          </div>
        </div>
      </div>

      <ScrollArea className="flex-1 p-4">
        {!result && (
          <div className="flex items-center justify-center h-full text-muted-foreground">
            <p>Click "Run" to execute the code</p>
          </div>
        )}

        {result && (
          <div className="space-y-4">
            {result.error && (
              <Card className="p-4 bg-destructive/10 border-destructive">
                <h4 className="text-sm font-semibold text-destructive mb-2">Error</h4>
                <pre className="text-sm text-destructive whitespace-pre-wrap">
                  {result.error}
                </pre>
              </Card>
            )}

            {result.output && !result.error && (
              <Card className="p-4">
                <h4 className="text-sm font-semibold text-foreground mb-2">Output</h4>
                <pre className="text-sm text-foreground whitespace-pre-wrap">
                  {result.output}
                </pre>
              </Card>
            )}

            {result.logs.length > 0 && (
              <Card className="p-4">
                <h4 className="text-sm font-semibold text-foreground mb-2">Console Logs</h4>
                <div className="space-y-1">
                  {result.logs.map((log, i) => (
                    <pre key={i} className="text-sm text-muted-foreground">
                      {log}
                    </pre>
                  ))}
                </div>
              </Card>
            )}
          </div>
        )}
      </ScrollArea>
    </div>
  );
};
