import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import {
  Sparkles,
  RefreshCw,
  CheckCircle,
  XCircle,
  Clock,
  TrendingUp,
  Undo2,
} from "lucide-react";
import { useSelfUpgrade } from "@/hooks/useSelfUpgrade";

export const SelfUpgradePanel = () => {
  const [activeTab, setActiveTab] = useState("suggestions");
  const {
    analyzeProject,
    applyUpgrade,
    rollbackUpgrade,
    dismissSuggestion,
    suggestions,
    history,
    analyzing,
    upgrading,
  } = useSelfUpgrade();

  const [selectedSuggestions, setSelectedSuggestions] = useState<string[]>([]);

  const toggleSuggestion = (id: string) => {
    setSelectedSuggestions(prev =>
      prev.includes(id) ? prev.filter(s => s !== id) : [...prev, id]
    );
  };

  const handleApplySelected = async () => {
    if (selectedSuggestions.length > 0) {
      await applyUpgrade(selectedSuggestions);
      setSelectedSuggestions([]);
    }
  };

  return (
    <div className="h-full flex flex-col bg-card">
      <div className="border-b border-border p-4">
        <div className="flex items-center gap-2 mb-2">
          <Sparkles className="h-5 w-5 text-primary" />
          <h2 className="text-lg font-semibold text-foreground">Self Upgrade Engine</h2>
        </div>
        <p className="text-sm text-muted-foreground">
          AI-powered code analysis and improvement suggestions
        </p>
      </div>

      <Tabs value={activeTab} onValueChange={setActiveTab} className="flex-1 flex flex-col">
        <TabsList className="w-full justify-start rounded-none border-b">
          <TabsTrigger value="suggestions" className="gap-2">
            <TrendingUp className="h-4 w-4" />
            Suggestions
            {suggestions.length > 0 && (
              <Badge variant="secondary" className="ml-1">
                {suggestions.length}
              </Badge>
            )}
          </TabsTrigger>
          <TabsTrigger value="history" className="gap-2">
            <Clock className="h-4 w-4" />
            History
          </TabsTrigger>
        </TabsList>

        <ScrollArea className="flex-1">
          <TabsContent value="suggestions" className="p-4 space-y-4 m-0">
            <div className="flex gap-2">
              <Button
                onClick={analyzeProject}
                disabled={analyzing}
                className="flex-1"
              >
                {analyzing ? (
                  <>
                    <RefreshCw className="mr-2 h-4 w-4 animate-spin" />
                    Analyzing...
                  </>
                ) : (
                  <>
                    <Sparkles className="mr-2 h-4 w-4" />
                    Analyze Project
                  </>
                )}
              </Button>
              {selectedSuggestions.length > 0 && (
                <Button
                  onClick={handleApplySelected}
                  disabled={upgrading}
                  variant="default"
                >
                  Apply ({selectedSuggestions.length})
                </Button>
              )}
            </div>

            {suggestions.length === 0 ? (
              <div className="text-center py-12 text-muted-foreground">
                <Sparkles className="h-12 w-12 mx-auto mb-3 opacity-50" />
                <p className="text-sm">No suggestions yet.</p>
                <p className="text-xs mt-1">Click "Analyze Project" to get started.</p>
              </div>
            ) : (
              <div className="space-y-3">
                {suggestions.map((suggestion) => (
                  <Card
                    key={suggestion.id}
                    className={`p-4 cursor-pointer transition-colors ${
                      selectedSuggestions.includes(suggestion.id)
                        ? "border-primary bg-primary/5"
                        : "hover:border-primary/50"
                    }`}
                    onClick={() => toggleSuggestion(suggestion.id)}
                  >
                    <div className="space-y-2">
                      <div className="flex items-start justify-between gap-2">
                        <div className="flex items-center gap-2 flex-wrap">
                          <Badge
                            variant={
                              suggestion.severity === "critical"
                                ? "destructive"
                                : suggestion.severity === "high"
                                ? "default"
                                : "secondary"
                            }
                          >
                            {suggestion.severity}
                          </Badge>
                          <Badge variant="outline">{suggestion.type}</Badge>
                          <span className="text-xs text-muted-foreground">
                            {suggestion.confidence}% confidence
                          </span>
                        </div>
                        <Button
                          size="sm"
                          variant="ghost"
                          onClick={(e) => {
                            e.stopPropagation();
                            dismissSuggestion(suggestion.id);
                          }}
                        >
                          <XCircle className="h-4 w-4" />
                        </Button>
                      </div>

                      <div>
                        <h4 className="text-sm font-medium text-foreground mb-1">
                          {suggestion.description}
                        </h4>
                        <p className="text-xs text-muted-foreground">
                          {suggestion.reasoning}
                        </p>
                      </div>

                      <div className="text-xs text-muted-foreground">
                        File: {suggestion.file}
                      </div>
                    </div>
                  </Card>
                ))}
              </div>
            )}
          </TabsContent>

          <TabsContent value="history" className="p-4 space-y-3 m-0">
            {history.length === 0 ? (
              <div className="text-center py-12 text-muted-foreground">
                <Clock className="h-12 w-12 mx-auto mb-3 opacity-50" />
                <p className="text-sm">No upgrade history yet.</p>
              </div>
            ) : (
              <div className="space-y-3">
                {history.map((entry) => (
                  <Card key={entry.id} className="p-4">
                    <div className="space-y-2">
                      <div className="flex items-center justify-between">
                        <div className="flex items-center gap-2">
                          <CheckCircle className="h-4 w-4 text-green-500" />
                          <span className="text-sm font-medium text-foreground">
                            {entry.suggestions.length} improvements applied
                          </span>
                        </div>
                        {entry.rollbackAvailable && (
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={() => rollbackUpgrade(entry.id)}
                          >
                            <Undo2 className="mr-2 h-4 w-4" />
                            Rollback
                          </Button>
                        )}
                      </div>

                      <p className="text-xs text-muted-foreground">
                        {new Date(entry.timestamp).toLocaleString()}
                      </p>

                      <div className="pt-2 space-y-1">
                        {entry.suggestions.map((s) => (
                          <div key={s.id} className="text-xs text-muted-foreground">
                            • {s.description}
                          </div>
                        ))}
                      </div>
                    </div>
                  </Card>
                ))}
              </div>
            )}
          </TabsContent>
        </ScrollArea>
      </Tabs>
    </div>
  );
};
