import { ReactNode } from "react";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { Button } from "@/components/ui/button";
import { useWallet } from "@/hooks/useWallet";
import { useNavigate } from "react-router-dom";
import { Wallet, AlertTriangle } from "lucide-react";

interface WalletGuardProps {
  children: ReactNode;
  requiredBalance?: number;
  onInsufficientBalance?: () => void;
}

export const WalletGuard = ({ 
  children, 
  requiredBalance = 1,
  onInsufficientBalance 
}: WalletGuardProps) => {
  const { wallet, isLoading } = useWallet();
  const navigate = useNavigate();

  if (isLoading) {
    return (
      <div className="flex items-center justify-center p-8">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
      </div>
    );
  }

  const hasBalance = (wallet?.balance || 0) >= requiredBalance;

  if (!hasBalance) {
    return (
      <Alert className="m-4">
        <AlertTriangle className="h-4 w-4" />
        <AlertTitle>Insufficient Balance</AlertTitle>
        <AlertDescription className="space-y-2">
          <p>
            You need at least ৳{requiredBalance.toFixed(2)} to use this feature.
            Your current balance: ৳{wallet?.balance.toFixed(2) || "0.00"}
          </p>
          <div className="flex gap-2">
            <Button
              size="sm"
              onClick={() => navigate("/studio")}
              className="gap-2"
            >
              <Wallet className="h-4 w-4" />
              Add Balance
            </Button>
            {onInsufficientBalance && (
              <Button size="sm" variant="outline" onClick={onInsufficientBalance}>
                Cancel
              </Button>
            )}
          </div>
        </AlertDescription>
      </Alert>
    );
  }

  return <>{children}</>;
};
