import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Code2, Package, Brain, Sparkles, Wallet, Zap, FileCode, Rocket } from "lucide-react";

interface WelcomeDashboardProps {
  onNavigate: (tab: "ide" | "modules" | "memory") => void;
}

export const WelcomeDashboard = ({ onNavigate }: WelcomeDashboardProps) => {
  const features = [
    {
      icon: Code2,
      title: "Full IDE",
      description: "Monaco editor, file explorer, live preview, multi-file tabs",
      action: () => onNavigate("ide"),
    },
    {
      icon: Sparkles,
      title: "AI Code Generator",
      description: "Generate code with AI, see diffs, apply changes automatically",
      action: () => onNavigate("ide"),
    },
    {
      icon: Package,
      title: "Module Generator",
      description: "Generate 50+ Smart Seba modules: Salon, Bazar, Hospital, ERP, etc.",
      action: () => onNavigate("modules"),
    },
    {
      icon: Brain,
      title: "Memory System",
      description: "LangCache-style long-term memory for preferences and instructions",
      action: () => onNavigate("memory"),
    },
  ];

  return (
    <div className="h-full overflow-auto p-8 bg-gradient-to-br from-background via-background to-muted/20">
      <div className="max-w-6xl mx-auto space-y-8">
        <div className="text-center space-y-4">
          <div className="flex items-center justify-center gap-3">
            <FileCode className="h-12 w-12 text-primary" />
            <h1 className="text-4xl font-bold text-foreground">Smart Friend Core Builder v1</h1>
          </div>
          <p className="text-xl text-muted-foreground max-w-2xl mx-auto">
            70-80% Lovable-style AI Code Builder for the Smart Seba Ecosystem
          </p>
          <div className="flex items-center justify-center gap-6 text-sm">
            <div className="flex items-center gap-2">
              <Zap className="h-4 w-4 text-primary" />
              <span className="text-foreground">AI-Powered</span>
            </div>
            <div className="flex items-center gap-2">
              <Rocket className="h-4 w-4 text-primary" />
              <span className="text-foreground">Multi-Framework</span>
            </div>
            <div className="flex items-center gap-2">
              <Wallet className="h-4 w-4 text-primary" />
              <span className="text-foreground">Credit System</span>
            </div>
          </div>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          {features.map((feature) => {
            const Icon = feature.icon;
            return (
              <Card
                key={feature.title}
                className="hover:border-primary transition-all cursor-pointer group"
                onClick={feature.action}
              >
                <CardHeader>
                  <div className="flex items-center gap-3">
                    <div className="p-3 rounded-lg bg-primary/10 group-hover:bg-primary/20 transition-colors">
                      <Icon className="h-6 w-6 text-primary" />
                    </div>
                    <div>
                      <CardTitle>{feature.title}</CardTitle>
                      <CardDescription className="mt-1">{feature.description}</CardDescription>
                    </div>
                  </div>
                </CardHeader>
              </Card>
            );
          })}
        </div>

        <Card className="bg-muted/30">
          <CardHeader>
            <CardTitle>What Can You Build?</CardTitle>
            <CardDescription>
              Full-stack applications with Laravel, React, Node.js, Flutter, and more
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
              {[
                "Smart Salon",
                "Smart Bazar",
                "Smart Hospital",
                "Smart Academy",
                "Smart ERP",
                "Smart Pay",
                "Smart News",
                "Smart Doctor",
                "Smart Lawyer",
                "Smart Marketer",
                "Smart Design",
                "Smart Tourist",
              ].map((module) => (
                <div
                  key={module}
                  className="p-3 rounded-lg bg-background border border-border hover:border-primary transition-colors"
                >
                  <span className="text-sm text-foreground">{module}</span>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-r from-primary/10 to-secondary/10 border-primary/20">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <h3 className="text-lg font-semibold text-foreground mb-2">Ready to Build?</h3>
                <p className="text-muted-foreground">
                  Start with the IDE or generate a complete module in one click
                </p>
              </div>
              <div className="flex gap-2">
                <Button onClick={() => onNavigate("ide")} className="gap-2">
                  <Code2 className="h-4 w-4" />
                  Open IDE
                </Button>
                <Button variant="outline" onClick={() => onNavigate("modules")} className="gap-2">
                  <Package className="h-4 w-4" />
                  Generate Module
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>

        <div className="text-center text-sm text-muted-foreground">
          <p>Smart Friend Core Builder v1.0 - Built for Smart Seba Ecosystem</p>
          <p className="mt-1">Supports: Laravel, React, Node.js, Flutter, PHP, HTML/CSS/JS</p>
        </div>
      </div>
    </div>
  );
};
