import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { useNavigate } from "react-router-dom";
import { LucideIcon } from "lucide-react";
import * as Icons from "lucide-react";

interface SebaModuleCardProps {
  slug: string;
  name: string;
  nameBn?: string;
  shortTagline?: string;
  shortTaglineBn?: string;
  icon?: string;
  category: string;
  status?: string;
  progressPercent?: number;
  isPublicUnlocked?: boolean;
  isOwner?: boolean;
}

export const SebaModuleCard = ({
  slug,
  name,
  nameBn,
  shortTagline,
  shortTaglineBn,
  icon,
  category,
  status,
  progressPercent,
  isPublicUnlocked,
  isOwner = false,
}: SebaModuleCardProps) => {
  const navigate = useNavigate();

  // Get icon component
  const getIcon = () => {
    if (icon && icon in Icons) {
      const IconComponent = Icons[icon as keyof typeof Icons] as LucideIcon;
      return <IconComponent className="w-8 h-8" />;
    }
    return <Icons.Box className="w-8 h-8" />;
  };

  // Get status badge
  const getStatusBadge = () => {
    if (isPublicUnlocked) {
      return <Badge variant="default" className="bg-success text-success-foreground">Available</Badge>;
    }
    if (status === "active" || status === "live") {
      return <Badge variant="default">Live</Badge>;
    }
    if (status === "development" || status === "in_progress") {
      return <Badge variant="secondary">In Development</Badge>;
    }
    return <Badge variant="secondary">Coming Soon</Badge>;
  };

  // Get category color
  const getCategoryColor = () => {
    const colors: Record<string, string> = {
      ai: "text-primary",
      education: "text-accent",
      health: "text-success",
      business: "text-warning",
      finance: "text-primary",
      ecommerce: "text-accent",
      content: "text-primary",
    };
    return colors[category.toLowerCase()] || "text-muted-foreground";
  };

  return (
    <Card className="group hover:shadow-lg transition-all duration-300 hover:-translate-y-1 border-border/50 hover:border-primary/30 bg-card">
      <CardHeader className="space-y-3">
        <div className="flex items-start justify-between">
          <div className={`p-3 rounded-xl bg-gradient-to-br from-primary/10 to-accent/10 ${getCategoryColor()} group-hover:scale-110 transition-transform`}>
            {getIcon()}
          </div>
          {getStatusBadge()}
        </div>

        <div className="space-y-1">
          <CardTitle className="text-xl font-bold group-hover:text-primary transition-colors line-clamp-1">
            {name}
          </CardTitle>
          {nameBn && (
            <div className="text-sm text-muted-foreground font-medium">
              {nameBn}
            </div>
          )}
        </div>
      </CardHeader>

      <CardContent className="space-y-4">
        <CardDescription className="line-clamp-2 text-sm leading-relaxed">
          {shortTagline || "AI-powered smart solution for modern needs"}
        </CardDescription>

        {isOwner && progressPercent !== undefined && (
          <div className="space-y-1">
            <div className="flex justify-between text-xs text-muted-foreground">
              <span>Progress</span>
              <span className="font-semibold">{progressPercent}%</span>
            </div>
            <div className="w-full h-1.5 bg-secondary rounded-full overflow-hidden">
              <div 
                className="h-full bg-gradient-to-r from-primary to-accent transition-all duration-300"
                style={{ width: `${progressPercent}%` }}
              />
            </div>
          </div>
        )}

        <Button 
          variant="outline" 
          className="w-full rounded-lg hover:bg-primary hover:text-primary-foreground transition-all"
          onClick={() => navigate(`/module/${slug}`)}
        >
          View Details
        </Button>
      </CardContent>
    </Card>
  );
};
