import { useQuery } from "@tanstack/react-query";
import { supabase } from "@/integrations/supabase/client";
import { SebaModuleCard } from "./SebaModuleCard";
import { useOwnerCheck } from "@/hooks/useOwnerCheck";
import { Skeleton } from "@/components/ui/skeleton";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { AlertCircle } from "lucide-react";

export const SebaModulesGrid = () => {
  const { isOwner } = useOwnerCheck();

  const { data: modules, isLoading, error } = useQuery({
    queryKey: ["seba-modules"],
    queryFn: async () => {
      const { data, error } = await supabase
        .from("seba_modules")
        .select("*")
        .eq("is_visible_public", true)
        .order("name", { ascending: true });

      if (error) throw error;
      return data;
    },
  });

  if (isLoading) {
    return (
      <section id="modules-section" className="py-16 md:py-24 bg-secondary/10">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <Skeleton className="h-10 w-64 mx-auto mb-4" />
            <Skeleton className="h-6 w-96 mx-auto" />
          </div>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
            {[...Array(8)].map((_, i) => (
              <Skeleton key={i} className="h-80 rounded-xl" />
            ))}
          </div>
        </div>
      </section>
    );
  }

  if (error) {
    return (
      <section id="modules-section" className="py-16 md:py-24 bg-secondary/10">
        <div className="container mx-auto px-4">
          <Alert variant="destructive">
            <AlertCircle className="h-4 w-4" />
            <AlertDescription>
              Failed to load modules. Please try again later.
            </AlertDescription>
          </Alert>
        </div>
      </section>
    );
  }

  if (!modules || modules.length === 0) {
    return (
      <section id="modules-section" className="py-16 md:py-24 bg-secondary/10">
        <div className="container mx-auto px-4">
          <div className="text-center">
            <p className="text-muted-foreground">No modules available yet.</p>
          </div>
        </div>
      </section>
    );
  }

  return (
    <section id="modules-section" className="py-16 md:py-24 bg-secondary/10">
      <div className="container mx-auto px-4">
        {/* Section Header */}
        <div className="text-center mb-12 space-y-4">
          <div className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-primary/10 border border-primary/20 mb-4">
            <span className="text-sm font-semibold text-primary">Our Ecosystem</span>
          </div>
          
          <h2 className="text-3xl md:text-4xl lg:text-5xl font-bold">
            Explore <span className="text-primary">50+ Smart Modules</span>
          </h2>
          
          <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
            Comprehensive AI-powered solutions for every aspect of modern life and business
          </p>
        </div>

        {/* Modules Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
          {modules.map((module) => (
            <SebaModuleCard
              key={module.id}
              slug={module.slug}
              name={module.name}
              nameBn={module.name_bn || undefined}
              shortTagline={module.short_tagline || undefined}
              shortTaglineBn={module.short_tagline_bn || undefined}
              icon={module.icon || undefined}
              category={module.category}
              status={module.status || undefined}
              progressPercent={module.progress_percent || undefined}
              isPublicUnlocked={module.is_public_unlocked || false}
              isOwner={isOwner}
            />
          ))}
        </div>
      </div>
    </section>
  );
};
