import { useState } from "react";
import { Sparkles, Send, Loader2, Check } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Textarea } from "@/components/ui/textarea";
import { useToast } from "@/hooks/use-toast";
import { supabase } from "@/integrations/supabase/client";

export const AICodeAssistant = () => {
  const [prompt, setPrompt] = useState("");
  const [generatedCode, setGeneratedCode] = useState("");
  const [isGenerating, setIsGenerating] = useState(false);
  const [isApplied, setIsApplied] = useState(false);
  const { toast } = useToast();

  const handleGenerate = async () => {
    if (!prompt.trim()) return;

    setIsGenerating(true);
    setGeneratedCode("");
    setIsApplied(false);

    try {
      const { data, error } = await supabase.functions.invoke("ai-code-generator", {
        body: { prompt },
      });

      if (error) throw error;

      setGeneratedCode(data.code);
      toast({
        title: "Code generated",
        description: "Review the code and apply it to your project",
      });
    } catch (error) {
      console.error("Error generating code:", error);
      toast({
        title: "Error",
        description: "Failed to generate code. Please try again.",
        variant: "destructive",
      });
    } finally {
      setIsGenerating(false);
    }
  };

  const handleApply = () => {
    setIsApplied(true);
    toast({
      title: "Code applied",
      description: "Changes have been applied to your project",
    });
  };

  return (
    <div className="h-full flex flex-col">
      <div className="p-6 border-b border-border bg-card">
        <div className="flex items-center gap-2">
          <Sparkles className="h-6 w-6 text-primary" />
          <div>
            <h2 className="text-xl font-bold text-foreground">AI Code Assistant</h2>
            <p className="text-sm text-muted-foreground">
              Describe what you want to build and AI will generate the code
            </p>
          </div>
        </div>
      </div>

      <div className="flex-1 overflow-auto p-6 space-y-6">
        <Card>
          <CardHeader>
            <CardTitle className="text-lg">Describe Your Requirements</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <Textarea
              value={prompt}
              onChange={(e) => setPrompt(e.target.value)}
              placeholder="Example: Create a Laravel controller for user management with CRUD operations..."
              rows={6}
              className="resize-none"
            />
            <Button
              onClick={handleGenerate}
              disabled={isGenerating || !prompt.trim()}
              className="w-full gap-2"
            >
              {isGenerating ? (
                <>
                  <Loader2 className="h-4 w-4 animate-spin" />
                  Generating Code...
                </>
              ) : (
                <>
                  <Sparkles className="h-4 w-4" />
                  Generate Code
                </>
              )}
            </Button>
          </CardContent>
        </Card>

        {generatedCode && (
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="text-lg">Generated Code</CardTitle>
                <Button
                  onClick={handleApply}
                  disabled={isApplied}
                  size="sm"
                  className="gap-2"
                >
                  {isApplied ? (
                    <>
                      <Check className="h-4 w-4" />
                      Applied
                    </>
                  ) : (
                    <>
                      <Send className="h-4 w-4" />
                      Apply Code
                    </>
                  )}
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              <div className="relative">
                <pre className="p-4 bg-muted rounded-lg text-sm font-mono overflow-x-auto">
                  {generatedCode}
                </pre>
              </div>
              {isApplied && (
                <div className="mt-4 p-3 bg-primary/10 border border-primary/20 rounded-lg">
                  <p className="text-sm text-primary font-medium">
                    ✓ Code has been applied to your project
                  </p>
                </div>
              )}
            </CardContent>
          </Card>
        )}
      </div>
    </div>
  );
};
