import { useState, useEffect } from "react";
import { Copy, Check } from "lucide-react";
import { Button } from "@/components/ui/button";
import { useToast } from "@/hooks/use-toast";

interface CodeViewerProps {
  selectedFile: string | null;
}

const fileMockData: Record<string, string> = {
  "src/App.tsx": `import { Toaster } from "@/components/ui/toaster";
import { Toaster as Sonner } from "@/components/ui/sonner";
import { TooltipProvider } from "@/components/ui/tooltip";
import { QueryClient, QueryClientProvider } from "@tanstack/react-query";
import { BrowserRouter, Routes, Route } from "react-router-dom";
import Index from "./pages/Index";

const queryClient = new QueryClient();

const App = () => (
  <QueryClientProvider client={queryClient}>
    <TooltipProvider>
      <Toaster />
      <Sonner />
      <BrowserRouter>
        <Routes>
          <Route path="/" element={<Index />} />
        </Routes>
      </BrowserRouter>
    </TooltipProvider>
  </QueryClientProvider>
);

export default App;`,
  "src/main.tsx": `import { createRoot } from 'react-dom/client'
import App from './App.tsx'
import './index.css'

createRoot(document.getElementById("root")!).render(<App />);`,
  "package.json": `{
  "name": "smart-friend-app",
  "private": true,
  "version": "0.0.0",
  "type": "module",
  "scripts": {
    "dev": "vite",
    "build": "vite build",
    "preview": "vite preview"
  },
  "dependencies": {
    "react": "^18.3.1",
    "react-dom": "^18.3.1",
    "react-router-dom": "^6.30.1"
  }
}`,
  "README.md": `# Smart Friend Application

Welcome to Smart Friend - Your AI-powered development assistant.

## Features
- AI Code Generation
- Module Builder
- Smart Wallet
- Subscription Management

## Getting Started
\`\`\`bash
npm install
npm run dev
\`\`\``,
};

export const CodeViewer = ({ selectedFile }: CodeViewerProps) => {
  const [copied, setCopied] = useState(false);
  const { toast } = useToast();
  const [code, setCode] = useState("");

  useEffect(() => {
    if (selectedFile) {
      setCode(fileMockData[selectedFile] || `// File: ${selectedFile}\n// Content not available in preview mode`);
    }
  }, [selectedFile]);

  const handleCopy = async () => {
    await navigator.clipboard.writeText(code);
    setCopied(true);
    toast({ title: "Copied to clipboard" });
    setTimeout(() => setCopied(false), 2000);
  };

  if (!selectedFile) {
    return (
      <div className="flex items-center justify-center h-full bg-background">
        <p className="text-muted-foreground">Select a file to view its contents</p>
      </div>
    );
  }

  const lines = code.split("\n");

  return (
    <div className="h-full flex flex-col bg-background">
      <div className="flex items-center justify-between p-3 border-b border-border bg-card">
        <div>
          <h3 className="text-sm font-semibold text-foreground">{selectedFile}</h3>
          <p className="text-xs text-muted-foreground">Read-only preview</p>
        </div>
        <Button
          size="sm"
          variant="outline"
          onClick={handleCopy}
          className="gap-2"
        >
          {copied ? <Check className="h-4 w-4" /> : <Copy className="h-4 w-4" />}
          {copied ? "Copied" : "Copy"}
        </Button>
      </div>
      <div className="flex-1 overflow-auto">
        <div className="flex">
          <div className="px-4 py-4 bg-muted/30 border-r border-border select-none">
            {lines.map((_, i) => (
              <div key={i} className="text-xs text-muted-foreground text-right leading-6">
                {i + 1}
              </div>
            ))}
          </div>
          <pre className="flex-1 p-4 text-sm font-mono leading-6 text-foreground overflow-x-auto">
            {code}
          </pre>
        </div>
      </div>
    </div>
  );
};
