import { useState } from "react";
import { ChevronRight, ChevronDown, File, Folder } from "lucide-react";
import { cn } from "@/lib/utils";

interface FileNode {
  name: string;
  type: "file" | "folder";
  path: string;
  children?: FileNode[];
}

const projectFiles: FileNode[] = [
  {
    name: "src",
    type: "folder",
    path: "src",
    children: [
      {
        name: "components",
        type: "folder",
        path: "src/components",
        children: [
          { name: "Button.tsx", type: "file", path: "src/components/Button.tsx" },
          { name: "Card.tsx", type: "file", path: "src/components/Card.tsx" },
        ],
      },
      {
        name: "pages",
        type: "folder",
        path: "src/pages",
        children: [
          { name: "Index.tsx", type: "file", path: "src/pages/Index.tsx" },
          { name: "Auth.tsx", type: "file", path: "src/pages/Auth.tsx" },
        ],
      },
      {
        name: "hooks",
        type: "folder",
        path: "src/hooks",
        children: [
          { name: "useAuth.tsx", type: "file", path: "src/hooks/useAuth.tsx" },
        ],
      },
      { name: "App.tsx", type: "file", path: "src/App.tsx" },
      { name: "main.tsx", type: "file", path: "src/main.tsx" },
      { name: "index.css", type: "file", path: "src/index.css" },
    ],
  },
  {
    name: "public",
    type: "folder",
    path: "public",
    children: [
      { name: "robots.txt", type: "file", path: "public/robots.txt" },
    ],
  },
  { name: "package.json", type: "file", path: "package.json" },
  { name: "vite.config.ts", type: "file", path: "vite.config.ts" },
  { name: "tsconfig.json", type: "file", path: "tsconfig.json" },
  { name: "README.md", type: "file", path: "README.md" },
];

interface FileTreeItemProps {
  node: FileNode;
  level: number;
  onFileSelect: (path: string) => void;
  selectedFile: string | null;
}

const FileTreeItem = ({ node, level, onFileSelect, selectedFile }: FileTreeItemProps) => {
  const [isExpanded, setIsExpanded] = useState(level === 0);

  const handleClick = () => {
    if (node.type === "folder") {
      setIsExpanded(!isExpanded);
    } else {
      onFileSelect(node.path);
    }
  };

  return (
    <div>
      <div
        className={cn(
          "flex items-center gap-2 px-2 py-1.5 cursor-pointer hover:bg-muted/50 rounded-md transition-colors",
          selectedFile === node.path && "bg-muted"
        )}
        style={{ paddingLeft: `${level * 12 + 8}px` }}
        onClick={handleClick}
      >
        {node.type === "folder" ? (
          <>
            {isExpanded ? (
              <ChevronDown className="h-4 w-4 text-muted-foreground" />
            ) : (
              <ChevronRight className="h-4 w-4 text-muted-foreground" />
            )}
            <Folder className="h-4 w-4 text-primary" />
          </>
        ) : (
          <>
            <div className="w-4" />
            <File className="h-4 w-4 text-muted-foreground" />
          </>
        )}
        <span className="text-sm text-foreground">{node.name}</span>
      </div>
      {node.type === "folder" && isExpanded && node.children && (
        <div>
          {node.children.map((child) => (
            <FileTreeItem
              key={child.path}
              node={child}
              level={level + 1}
              onFileSelect={onFileSelect}
              selectedFile={selectedFile}
            />
          ))}
        </div>
      )}
    </div>
  );
};

interface EnhancedFileTreeProps {
  onFileSelect: (path: string) => void;
  selectedFile: string | null;
}

export const EnhancedFileTree = ({ onFileSelect, selectedFile }: EnhancedFileTreeProps) => {
  return (
    <div className="h-full overflow-auto bg-card border-r border-border">
      <div className="p-3 border-b border-border">
        <h3 className="text-sm font-semibold text-foreground">Project Files</h3>
      </div>
      <div className="p-2">
        {projectFiles.map((node) => (
          <FileTreeItem
            key={node.path}
            node={node}
            level={0}
            onFileSelect={onFileSelect}
            selectedFile={selectedFile}
          />
        ))}
      </div>
    </div>
  );
};
