import { Check, Zap } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { useSubscription } from "@/hooks/useSubscription";
import { Badge } from "@/components/ui/badge";

const plans = [
  {
    name: "Basic",
    price: "Free",
    features: [
      "5 AI requests per day",
      "Basic module templates",
      "Community support",
      "1 project",
    ],
  },
  {
    name: "Pro",
    price: "৳999/mo",
    features: [
      "100 AI requests per day",
      "All module templates",
      "Priority support",
      "10 projects",
      "Custom domains",
    ],
    popular: true,
  },
  {
    name: "Unlimited",
    price: "৳2999/mo",
    features: [
      "Unlimited AI requests",
      "All Pro features",
      "24/7 support",
      "Unlimited projects",
      "White-label options",
      "API access",
    ],
  },
];

export const SubscriptionPlans = () => {
  const { subscription, isLoading, activatePlan } = useSubscription();

  const handleActivate = async (plan: string) => {
    await activatePlan.mutateAsync(plan);
  };

  return (
    <div className="h-full overflow-auto p-6">
      <div className="max-w-6xl mx-auto space-y-8">
        <div className="text-center">
          <h2 className="text-3xl font-bold text-foreground mb-2">Choose Your Plan</h2>
          <p className="text-muted-foreground">
            Unlock powerful features to build better applications faster
          </p>
          {subscription && (
            <Badge variant="default" className="mt-4">
              Current Plan: {subscription.plan}
              {subscription.is_active && " (Active)"}
            </Badge>
          )}
        </div>

        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          {plans.map((plan) => (
            <Card
              key={plan.name}
              className={plan.popular ? "border-primary shadow-lg relative" : ""}
            >
              {plan.popular && (
                <div className="absolute -top-3 left-1/2 -translate-x-1/2">
                  <Badge className="gap-1">
                    <Zap className="h-3 w-3" />
                    Most Popular
                  </Badge>
                </div>
              )}
              <CardHeader>
                <CardTitle className="text-2xl">{plan.name}</CardTitle>
                <CardDescription>
                  <span className="text-3xl font-bold text-foreground">{plan.price}</span>
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <ul className="space-y-2">
                  {plan.features.map((feature, i) => (
                    <li key={i} className="flex items-center gap-2 text-sm">
                      <Check className="h-4 w-4 text-primary" />
                      <span className="text-foreground">{feature}</span>
                    </li>
                  ))}
                </ul>
                <Button
                  className="w-full"
                  variant={plan.popular ? "default" : "outline"}
                  onClick={() => handleActivate(plan.name)}
                  disabled={
                    isLoading ||
                    activatePlan.isPending ||
                    subscription?.plan === plan.name
                  }
                >
                  {subscription?.plan === plan.name
                    ? "Current Plan"
                    : `Activate ${plan.name}`}
                </Button>
              </CardContent>
            </Card>
          ))}
        </div>

        <Card className="bg-muted/30">
          <CardContent className="pt-6">
            <div className="flex items-center justify-between">
              <div>
                <h3 className="font-semibold text-foreground mb-1">Need a custom plan?</h3>
                <p className="text-sm text-muted-foreground">
                  Contact us for enterprise solutions
                </p>
              </div>
              <Button variant="outline">Contact Sales</Button>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
};
