import { useEffect, useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { supabase } from "@/integrations/supabase/client";
import { LineChart, Line, BarChart, Bar, PieChart, Pie, Cell, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer } from "recharts";
import { Activity, CheckCircle2, Code, TrendingUp } from "lucide-react";
import { Skeleton } from "@/components/ui/skeleton";

interface AnalyticsData {
  testCoverage: { date: string; count: number; approved: number }[];
  fixSuccessRates: { status: string; count: number }[];
  patchHistory: { date: string; applied: number; pending: number; rejected: number }[];
  errorTrends: { date: string; high: number; medium: number; low: number }[];
  totalStats: {
    totalTests: number;
    approvedTests: number;
    totalPatches: number;
    appliedPatches: number;
    totalErrors: number;
    resolvedErrors: number;
  };
}

const COLORS = {
  primary: "hsl(var(--primary))",
  success: "hsl(var(--success))",
  warning: "hsl(var(--warning))",
  destructive: "hsl(var(--destructive))",
  muted: "hsl(var(--muted))",
};

export const CoreBrainAnalytics = () => {
  const [loading, setLoading] = useState(true);
  const [analytics, setAnalytics] = useState<AnalyticsData>({
    testCoverage: [],
    fixSuccessRates: [],
    patchHistory: [],
    errorTrends: [],
    totalStats: {
      totalTests: 0,
      approvedTests: 0,
      totalPatches: 0,
      appliedPatches: 0,
      totalErrors: 0,
      resolvedErrors: 0,
    },
  });

  useEffect(() => {
    fetchAnalytics();
  }, []);

  const fetchAnalytics = async () => {
    try {
      setLoading(true);

      // Fetch test data
      const { data: tests } = await supabase
        .from("generated_tests")
        .select("created_at, status, applied_at")
        .order("created_at", { ascending: true });

      // Fetch patch data
      const { data: patches } = await supabase
        .from("code_patches")
        .select("created_at, status, applied_at")
        .order("created_at", { ascending: true });

      // Fetch fix plans data
      const { data: fixPlans } = await supabase
        .from("fix_plans")
        .select("status")
        .order("created_at", { ascending: true });

      // Fetch scan results data
      const { data: scanResults } = await supabase
        .from("scan_results")
        .select("created_at, severity, resolved")
        .order("created_at", { ascending: true });

      // Process test coverage over time
      const testsByDate = processTestCoverage(tests || []);

      // Process fix success rates
      const fixRates = processFixSuccessRates(fixPlans || []);

      // Process patch history
      const patchData = processPatchHistory(patches || []);

      // Process error trends
      const errorData = processErrorTrends(scanResults || []);

      // Calculate total stats
      const totalStats = {
        totalTests: tests?.length || 0,
        approvedTests: tests?.filter((t) => t.status === "approved").length || 0,
        totalPatches: patches?.length || 0,
        appliedPatches: patches?.filter((p) => p.status === "applied").length || 0,
        totalErrors: scanResults?.length || 0,
        resolvedErrors: scanResults?.filter((e) => e.resolved).length || 0,
      };

      setAnalytics({
        testCoverage: testsByDate,
        fixSuccessRates: fixRates,
        patchHistory: patchData,
        errorTrends: errorData,
        totalStats,
      });
    } catch (error) {
      console.error("Error fetching analytics:", error);
    } finally {
      setLoading(false);
    }
  };

  const processTestCoverage = (tests: any[]) => {
    const dateMap = new Map<string, { count: number; approved: number }>();
    
    tests.forEach((test) => {
      const date = new Date(test.created_at).toLocaleDateString();
      const current = dateMap.get(date) || { count: 0, approved: 0 };
      current.count += 1;
      if (test.status === "approved") current.approved += 1;
      dateMap.set(date, current);
    });

    return Array.from(dateMap.entries())
      .map(([date, data]) => ({ date, ...data }))
      .slice(-30);
  };

  const processFixSuccessRates = (fixPlans: any[]) => {
    const statusMap = new Map<string, number>();
    
    fixPlans.forEach((plan) => {
      const count = statusMap.get(plan.status) || 0;
      statusMap.set(plan.status, count + 1);
    });

    return Array.from(statusMap.entries()).map(([status, count]) => ({
      status: status.charAt(0).toUpperCase() + status.slice(1),
      count,
    }));
  };

  const processPatchHistory = (patches: any[]) => {
    const dateMap = new Map<string, { applied: number; pending: number; rejected: number }>();
    
    patches.forEach((patch) => {
      const date = new Date(patch.created_at).toLocaleDateString();
      const current = dateMap.get(date) || { applied: 0, pending: 0, rejected: 0 };
      
      if (patch.status === "applied") current.applied += 1;
      else if (patch.status === "pending") current.pending += 1;
      else if (patch.status === "rejected") current.rejected += 1;
      
      dateMap.set(date, current);
    });

    return Array.from(dateMap.entries())
      .map(([date, data]) => ({ date, ...data }))
      .slice(-30);
  };

  const processErrorTrends = (scanResults: any[]) => {
    const dateMap = new Map<string, { high: number; medium: number; low: number }>();
    
    scanResults.forEach((result) => {
      const date = new Date(result.created_at).toLocaleDateString();
      const current = dateMap.get(date) || { high: 0, medium: 0, low: 0 };
      
      if (result.severity === "high") current.high += 1;
      else if (result.severity === "medium") current.medium += 1;
      else if (result.severity === "low") current.low += 1;
      
      dateMap.set(date, current);
    });

    return Array.from(dateMap.entries())
      .map(([date, data]) => ({ date, ...data }))
      .slice(-30);
  };

  if (loading) {
    return (
      <div className="space-y-4">
        <div className="grid gap-4 md:grid-cols-4">
          {[...Array(4)].map((_, i) => (
            <Card key={i}>
              <CardHeader className="space-y-2">
                <Skeleton className="h-4 w-24" />
                <Skeleton className="h-8 w-16" />
              </CardHeader>
            </Card>
          ))}
        </div>
        <div className="grid gap-4 md:grid-cols-2">
          {[...Array(4)].map((_, i) => (
            <Card key={i}>
              <CardHeader>
                <Skeleton className="h-6 w-32" />
              </CardHeader>
              <CardContent>
                <Skeleton className="h-64 w-full" />
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  }

  const { totalStats } = analytics;

  return (
    <div className="space-y-6">
      {/* Stats Overview */}
      <div className="grid gap-4 md:grid-cols-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Test Coverage</CardTitle>
            <CheckCircle2 className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{totalStats.approvedTests}/{totalStats.totalTests}</div>
            <p className="text-xs text-muted-foreground">
              {totalStats.totalTests > 0 ? Math.round((totalStats.approvedTests / totalStats.totalTests) * 100) : 0}% approved
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Patch Success</CardTitle>
            <Code className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{totalStats.appliedPatches}/{totalStats.totalPatches}</div>
            <p className="text-xs text-muted-foreground">
              {totalStats.totalPatches > 0 ? Math.round((totalStats.appliedPatches / totalStats.totalPatches) * 100) : 0}% applied
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Error Resolution</CardTitle>
            <Activity className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{totalStats.resolvedErrors}/{totalStats.totalErrors}</div>
            <p className="text-xs text-muted-foreground">
              {totalStats.totalErrors > 0 ? Math.round((totalStats.resolvedErrors / totalStats.totalErrors) * 100) : 0}% resolved
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Overall Health</CardTitle>
            <TrendingUp className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {totalStats.totalTests + totalStats.totalPatches + totalStats.totalErrors > 0
                ? Math.round(
                    ((totalStats.approvedTests + totalStats.appliedPatches + totalStats.resolvedErrors) /
                      (totalStats.totalTests + totalStats.totalPatches + totalStats.totalErrors)) *
                      100
                  )
                : 0}%
            </div>
            <p className="text-xs text-muted-foreground">System efficiency</p>
          </CardContent>
        </Card>
      </div>

      {/* Charts */}
      <div className="grid gap-4 md:grid-cols-2">
        {/* Test Coverage Over Time */}
        <Card>
          <CardHeader>
            <CardTitle>Test Coverage Trend</CardTitle>
            <CardDescription>Generated vs Approved tests over time</CardDescription>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <LineChart data={analytics.testCoverage}>
                <CartesianGrid strokeDasharray="3 3" stroke="hsl(var(--border))" />
                <XAxis dataKey="date" stroke="hsl(var(--muted-foreground))" fontSize={12} />
                <YAxis stroke="hsl(var(--muted-foreground))" fontSize={12} />
                <Tooltip
                  contentStyle={{
                    backgroundColor: "hsl(var(--popover))",
                    border: "1px solid hsl(var(--border))",
                    borderRadius: "var(--radius)",
                  }}
                />
                <Legend />
                <Line type="monotone" dataKey="count" stroke={COLORS.primary} name="Generated" strokeWidth={2} />
                <Line type="monotone" dataKey="approved" stroke={COLORS.success} name="Approved" strokeWidth={2} />
              </LineChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>

        {/* Fix Success Rates */}
        <Card>
          <CardHeader>
            <CardTitle>Fix Plan Status Distribution</CardTitle>
            <CardDescription>Current status of all fix plans</CardDescription>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <PieChart>
                <Pie
                  data={analytics.fixSuccessRates}
                  cx="50%"
                  cy="50%"
                  labelLine={false}
                  label={({ status, percent }) => `${status}: ${(percent * 100).toFixed(0)}%`}
                  outerRadius={80}
                  fill={COLORS.primary}
                  dataKey="count"
                >
                  {analytics.fixSuccessRates.map((entry, index) => (
                    <Cell
                      key={`cell-${index}`}
                      fill={
                        entry.status === "Approved"
                          ? COLORS.success
                          : entry.status === "Rejected"
                          ? COLORS.destructive
                          : COLORS.warning
                      }
                    />
                  ))}
                </Pie>
                <Tooltip
                  contentStyle={{
                    backgroundColor: "hsl(var(--popover))",
                    border: "1px solid hsl(var(--border))",
                    borderRadius: "var(--radius)",
                  }}
                />
              </PieChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>

        {/* Patch Application History */}
        <Card>
          <CardHeader>
            <CardTitle>Patch Application History</CardTitle>
            <CardDescription>Daily patch status breakdown</CardDescription>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <BarChart data={analytics.patchHistory}>
                <CartesianGrid strokeDasharray="3 3" stroke="hsl(var(--border))" />
                <XAxis dataKey="date" stroke="hsl(var(--muted-foreground))" fontSize={12} />
                <YAxis stroke="hsl(var(--muted-foreground))" fontSize={12} />
                <Tooltip
                  contentStyle={{
                    backgroundColor: "hsl(var(--popover))",
                    border: "1px solid hsl(var(--border))",
                    borderRadius: "var(--radius)",
                  }}
                />
                <Legend />
                <Bar dataKey="applied" fill={COLORS.success} name="Applied" />
                <Bar dataKey="pending" fill={COLORS.warning} name="Pending" />
                <Bar dataKey="rejected" fill={COLORS.destructive} name="Rejected" />
              </BarChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>

        {/* Error Trends */}
        <Card>
          <CardHeader>
            <CardTitle>Error Severity Trends</CardTitle>
            <CardDescription>Daily error distribution by severity</CardDescription>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <LineChart data={analytics.errorTrends}>
                <CartesianGrid strokeDasharray="3 3" stroke="hsl(var(--border))" />
                <XAxis dataKey="date" stroke="hsl(var(--muted-foreground))" fontSize={12} />
                <YAxis stroke="hsl(var(--muted-foreground))" fontSize={12} />
                <Tooltip
                  contentStyle={{
                    backgroundColor: "hsl(var(--popover))",
                    border: "1px solid hsl(var(--border))",
                    borderRadius: "var(--radius)",
                  }}
                />
                <Legend />
                <Line type="monotone" dataKey="high" stroke={COLORS.destructive} name="High" strokeWidth={2} />
                <Line type="monotone" dataKey="medium" stroke={COLORS.warning} name="Medium" strokeWidth={2} />
                <Line type="monotone" dataKey="low" stroke={COLORS.success} name="Low" strokeWidth={2} />
              </LineChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>
      </div>
    </div>
  );
};
