import { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Brain, Activity, Database, Zap, Clock, TrendingUp } from "lucide-react";
import LogStream from "./logStream";
import ErrorList from "./errorList";
import CriticalNotifications from "./criticalNotifications";
import { toast } from "sonner";
import { supabase } from "@/integrations/supabase/client";

const Dashboard = () => {
  const [scanning, setScanning] = useState(false);

  const handleGenerateFixPlan = async (errorId: string) => {
    try {
      toast.info('ফিক্স প্ল্যান তৈরি হচ্ছে... • Generating fix plan...');
      
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error('Not authenticated');

      const { data: error } = await supabase
        .from('scan_results')
        .select('*')
        .eq('id', errorId)
        .single();

      if (!error) throw new Error('Error not found');

      // Call the analyze function
      const { data, error: functionError } = await supabase.functions.invoke('core-brain-analyze', {
        body: {
          type: 'fix',
          context: {
            error_id: errorId,
            file_path: error.file_path,
            issue_type: error.issue_type,
            message: error.message,
            severity: error.severity
          }
        }
      });

      if (functionError) throw functionError;

      // Create fix plan
      const { error: insertError } = await supabase
        .from('fix_plans')
        .insert({
          title: `Fix: ${error.message.substring(0, 50)}...`,
          description: data?.suggestion || 'AI-generated fix plan',
          priority: error.severity === 'high' || error.severity === 'critical' ? 'high' : 'medium',
          status: 'pending',
          ai_suggestion: data?.suggestion,
          scan_result_id: errorId,
          created_by: user.id
        });

      if (insertError) throw insertError;

      toast.success('ফিক্স প্ল্যান তৈরি সম্পন্ন! • Fix plan generated successfully!');
    } catch (error) {
      console.error('Error generating fix plan:', error);
      toast.error('ফিক্স প্ল্যান তৈরি ব্যর্থ • Failed to generate fix plan');
    }
  };

  const handleQuickScan = async () => {
    setScanning(true);
    toast.info('প্রজেক্ট স্ক্যান শুরু... • Starting project scan...');

    try {
      const { data, error } = await supabase.functions.invoke('core-brain-scanner', {
        body: { scan_type: 'quick' }
      });

      if (error) throw error;

      await supabase.from('core_brain_logs').insert({
        type: 'info',
        message: 'Quick scan completed',
        details: data
      });

      toast.success('স্ক্যান সম্পন্ন! • Scan completed!');
    } catch (error) {
      console.error('Scan error:', error);
      toast.error('স্ক্যান ব্যর্থ • Scan failed');
    } finally {
      setScanning(false);
    }
  };

  const systemStats = [
    {
      title: "সিস্টেম স্বাস্থ্য",
      titleEn: "System Health",
      value: "99.9%",
      icon: Activity,
      color: "text-green-500"
    },
    {
      title: "ডেটাবেস",
      titleEn: "Database",
      value: "সক্রিয়",
      icon: Database,
      color: "text-blue-500"
    },
    {
      title: "স্বয়ংক্রিয় স্ক্যান",
      titleEn: "Auto Scan",
      value: "প্রতি ৬ ঘণ্টা",
      icon: Clock,
      color: "text-purple-500"
    },
    {
      title: "কর্মক্ষমতা",
      titleEn: "Performance",
      value: "চমৎকার",
      icon: TrendingUp,
      color: "text-orange-500"
    }
  ];

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-4">
        <div>
          <h1 className="text-3xl font-bold flex items-center gap-3">
            <Brain className="w-8 h-8 text-primary" />
            Smart Friend Core Brain
          </h1>
          <p className="text-muted-foreground mt-1">
            আপনার সিস্টেম এখন রিয়েলটাইম মনিটরিং এর অধীনে
          </p>
        </div>
        <Button onClick={handleQuickScan} disabled={scanning} className="gap-2">
          <Zap className="w-4 h-4" />
          {scanning ? 'স্ক্যান চলছে...' : 'দ্রুত স্ক্যান • Quick Scan'}
        </Button>
      </div>

      {/* System Stats */}
      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
        {systemStats.map((stat) => {
          const Icon = stat.icon;
          return (
            <Card key={stat.titleEn}>
              <CardHeader className="pb-2">
                <CardDescription className="flex items-center gap-2">
                  <Icon className={`w-4 h-4 ${stat.color}`} />
                  {stat.title}
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{stat.value}</div>
                <p className="text-xs text-muted-foreground">{stat.titleEn}</p>
              </CardContent>
            </Card>
          );
        })}
      </div>

      {/* Main Content Grid */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Critical Notifications */}
        <CriticalNotifications />

        {/* Log Stream */}
        <LogStream />

        {/* Error List */}
        <ErrorList onGenerateFixPlan={handleGenerateFixPlan} />
      </div>
    </div>
  );
};

export default Dashboard;
