import { useEffect, useState } from "react";
import { supabase } from "@/integrations/supabase/client";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { AlertCircle, CheckCircle, Sparkles, TestTube2 } from "lucide-react";
import type { Json } from "@/integrations/supabase/types";
import { toast } from "sonner";
import { TestGenerator } from "./testGenerator";

type ErrorEntry = {
  id: string;
  scan_type: string;
  file_path: string | null;
  issue_type: string;
  severity: string;
  message: string;
  details: Json | null;
  resolved: boolean;
  created_at: string;
};

interface ErrorListProps {
  onGenerateFixPlan: (errorId: string) => void;
}

const ErrorList = ({ onGenerateFixPlan }: ErrorListProps) => {
  const [errors, setErrors] = useState<ErrorEntry[]>([]);
  const [loading, setLoading] = useState(true);
  const [generatingTest, setGeneratingTest] = useState<string | null>(null);

  useEffect(() => {
    fetchErrors();
    const channel = subscribeToErrors();

    return () => {
      supabase.removeChannel(channel);
    };
  }, []);

  const fetchErrors = async () => {
    try {
      const { data, error } = await supabase
        .from('scan_results')
        .select('*')
        .eq('resolved', false)
        .order('created_at', { ascending: false })
        .limit(20);

      if (error) throw error;
      setErrors(data || []);
    } catch (error) {
      console.error('Error fetching errors:', error);
      toast.error('Failed to load errors');
    } finally {
      setLoading(false);
    }
  };

  const subscribeToErrors = () => {
    return supabase
      .channel('error_list_updates')
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'scan_results'
        },
        () => {
          fetchErrors();
        }
      )
      .subscribe();
  };

  const handleResolve = async (errorId: string) => {
    try {
      const { error } = await supabase
        .from('scan_results')
        .update({ resolved: true })
        .eq('id', errorId);

      if (error) throw error;
      toast.success('ত্রুটি সমাধান হিসেবে চিহ্নিত • Error marked as resolved');
    } catch (error) {
      console.error('Error resolving:', error);
      toast.error('Failed to resolve error');
    }
  };

  const handleGenerateTest = async (error: ErrorEntry) => {
    setGeneratingTest(error.id);
    await TestGenerator.generateFromError({
      errorId: error.id,
      errorMessage: error.message,
      filePath: error.file_path,
      issueType: error.issue_type,
      severity: error.severity,
      details: error.details
    });
    setGeneratingTest(null);
  };

  const getSeverityColor = (severity: string) => {
    switch (severity.toLowerCase()) {
      case 'critical':
      case 'high':
        return 'destructive';
      case 'medium':
        return 'secondary';
      case 'low':
        return 'outline';
      default:
        return 'default';
    }
  };

  if (loading) {
    return (
      <Card>
        <CardContent className="py-8 text-center text-muted-foreground">
          লোড হচ্ছে... • Loading...
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <AlertCircle className="w-5 h-5 text-red-500" />
          সনাক্ত ত্রুটি • Detected Errors
          {errors.length > 0 && (
            <Badge variant="destructive" className="ml-2">
              {errors.length}
            </Badge>
          )}
        </CardTitle>
      </CardHeader>
      <CardContent>
        {errors.length === 0 ? (
          <div className="text-center py-8">
            <CheckCircle className="w-12 h-12 text-green-500 mx-auto mb-3" />
            <p className="text-lg font-medium">কোনো ত্রুটি নেই!</p>
            <p className="text-sm text-muted-foreground">No errors detected</p>
          </div>
        ) : (
          <div className="space-y-3">
            {errors.map((error) => (
              <div
                key={error.id}
                className="p-4 rounded-lg border bg-card space-y-3"
              >
                <div className="flex items-start justify-between gap-2">
                  <div className="flex-1 space-y-2">
                    <div className="flex items-center gap-2 flex-wrap">
                      <Badge variant={getSeverityColor(error.severity)}>
                        {error.severity}
                      </Badge>
                      <Badge variant="outline">{error.issue_type}</Badge>
                      {error.file_path && (
                        <span className="text-xs text-muted-foreground font-mono">
                          {error.file_path}
                        </span>
                      )}
                    </div>
                    <p className="text-sm font-medium">{error.message}</p>
                    {error.details && (
                      <pre className="text-xs text-muted-foreground bg-muted/50 p-2 rounded overflow-x-auto">
                        {JSON.stringify(error.details, null, 2)}
                      </pre>
                    )}
                    <p className="text-xs text-muted-foreground">
                      {new Date(error.created_at).toLocaleString('bn-BD')}
                    </p>
                  </div>
                </div>
                <div className="flex gap-2">
                  <Button
                    size="sm"
                    variant="outline"
                    onClick={() => onGenerateFixPlan(error.id)}
                    className="gap-2"
                  >
                    <Sparkles className="w-4 h-4" />
                    ফিক্স প্ল্যান তৈরি করুন • Generate Fix Plan
                  </Button>
                  <Button
                    size="sm"
                    variant="secondary"
                    onClick={() => handleGenerateTest(error)}
                    disabled={generatingTest === error.id}
                    className="gap-2"
                  >
                    <TestTube2 className="w-4 h-4" />
                    {generatingTest === error.id ? 'Generating...' : 'টেস্ট তৈরি করুন • Generate Test'}
                  </Button>
                  <Button
                    size="sm"
                    variant="secondary"
                    onClick={() => handleResolve(error.id)}
                  >
                    <CheckCircle className="w-4 h-4 mr-2" />
                    সমাধান চিহ্নিত করুন • Mark Resolved
                  </Button>
                </div>
              </div>
            ))}
          </div>
        )}
      </CardContent>
    </Card>
  );
};

export default ErrorList;
