import { supabase } from "@/integrations/supabase/client";
import { toast } from "sonner";

export interface FixPlanContext {
  errorId?: string;
  errorType?: string;
  errorMessage?: string;
  filePath?: string;
  severity?: string;
  customDescription?: string;
}

export class FixPlanGenerator {
  /**
   * Generate a fix plan from an error
   */
  static async generateFromError(context: FixPlanContext): Promise<string | null> {
    try {
      toast.info('AI ফিক্স প্ল্যান তৈরি করছে... • AI is generating fix plan...');

      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error('Not authenticated');

      // Call AI to analyze and generate fix suggestion
      const { data, error } = await supabase.functions.invoke('core-brain-analyze', {
        body: {
          type: 'fix',
          context: {
            error_id: context.errorId,
            error_type: context.errorType,
            message: context.errorMessage,
            file_path: context.filePath,
            severity: context.severity
          }
        }
      });

      if (error) throw error;

      // Determine priority based on severity
      let priority = 'medium';
      if (context.severity === 'critical' || context.severity === 'high') {
        priority = 'high';
      } else if (context.severity === 'low') {
        priority = 'low';
      }

      // Create the fix plan
      const { data: fixPlan, error: insertError } = await supabase
        .from('fix_plans')
        .insert({
          title: data?.title || `Fix: ${context.errorMessage?.substring(0, 50)}...`,
          description: data?.description || context.customDescription || 'AI-generated fix plan',
          priority,
          status: 'pending',
          ai_suggestion: data?.suggestion,
          scan_result_id: context.errorId || null,
          created_by: user.id
        })
        .select()
        .single();

      if (insertError) throw insertError;

      // Log the creation
      await supabase.from('core_brain_logs').insert({
        type: 'info',
        message: `Fix plan generated: ${fixPlan.title}`,
        details: { 
          fix_plan_id: fixPlan.id, 
          error_id: context.errorId,
          priority 
        },
        user_id: user.id
      });

      toast.success('ফিক্স প্ল্যান তৈরি সম্পন্ন! • Fix plan generated!');
      return fixPlan.id;
    } catch (error) {
      console.error('Error generating fix plan:', error);
      toast.error('ফিক্স প্ল্যান তৈরি ব্যর্থ • Failed to generate fix plan');
      return null;
    }
  }

  /**
   * Generate a custom fix plan without an error
   */
  static async generateCustom(title: string, description: string): Promise<string | null> {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error('Not authenticated');

      // Call AI to enhance the description
      const { data, error } = await supabase.functions.invoke('core-brain-analyze', {
        body: {
          type: 'enhancement',
          context: { title, description }
        }
      });

      if (error) {
        console.warn('AI enhancement failed, using original description');
      }

      const { data: fixPlan, error: insertError } = await supabase
        .from('fix_plans')
        .insert({
          title,
          description,
          priority: 'medium',
          status: 'pending',
          ai_suggestion: data?.suggestion || null,
          scan_result_id: null,
          created_by: user.id
        })
        .select()
        .single();

      if (insertError) throw insertError;

      await supabase.from('core_brain_logs').insert({
        type: 'info',
        message: `Custom fix plan created: ${title}`,
        details: { fix_plan_id: fixPlan.id },
        user_id: user.id
      });

      toast.success('কাস্টম ফিক্স প্ল্যান তৈরি! • Custom fix plan created!');
      return fixPlan.id;
    } catch (error) {
      console.error('Error creating custom fix plan:', error);
      toast.error('ফিক্স প্ল্যান তৈরি ব্যর্থ • Failed to create fix plan');
      return null;
    }
  }

  /**
   * Approve a fix plan
   */
  static async approvePlan(planId: string): Promise<boolean> {
    try {
      const { error } = await supabase
        .from('fix_plans')
        .update({ status: 'approved' })
        .eq('id', planId);

      if (error) throw error;

      await supabase.from('core_brain_logs').insert({
        type: 'success',
        message: 'Fix plan approved',
        details: { fix_plan_id: planId }
      });

      toast.success('ফিক্স প্ল্যান অনুমোদিত! • Fix plan approved!');
      return true;
    } catch (error) {
      console.error('Error approving plan:', error);
      toast.error('অনুমোদন ব্যর্থ • Approval failed');
      return false;
    }
  }

  /**
   * Reject a fix plan
   */
  static async rejectPlan(planId: string): Promise<boolean> {
    try {
      const { error } = await supabase
        .from('fix_plans')
        .update({ status: 'rejected' })
        .eq('id', planId);

      if (error) throw error;

      await supabase.from('core_brain_logs').insert({
        type: 'warning',
        message: 'Fix plan rejected',
        details: { fix_plan_id: planId }
      });

      toast.info('ফিক্স প্ল্যান প্রত্যাখ্যাত • Fix plan rejected');
      return true;
    } catch (error) {
      console.error('Error rejecting plan:', error);
      toast.error('প্রত্যাখ্যান ব্যর্থ • Rejection failed');
      return false;
    }
  }
}
