import { useState, useEffect } from "react";
import { Card } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { TestGenerator } from "./testGenerator";
import { supabase } from "@/integrations/supabase/client";
import { TestTube2, CheckCircle2, XCircle, Eye, FileCode } from "lucide-react";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription, DialogFooter } from "@/components/ui/dialog";
import { format } from "date-fns";

type TestStatus = 'pending' | 'approved' | 'rejected';

interface GeneratedTest {
  id: string;
  test_file_path: string;
  test_code: string;
  test_framework: string;
  description: string | null;
  status: TestStatus;
  created_at: string;
  applied_at: string | null;
  scan_result_id: string | null;
  scan_results?: {
    message: string;
    issue_type: string;
  };
}

export default function GeneratedTests() {
  const [tests, setTests] = useState<GeneratedTest[]>([]);
  const [loading, setLoading] = useState(true);
  const [processing, setProcessing] = useState<string | null>(null);
  const [previewTest, setPreviewTest] = useState<GeneratedTest | null>(null);

  const fetchTests = async () => {
    setLoading(true);
    const data = await TestGenerator.getPendingTests();
    setTests(data as GeneratedTest[]);
    setLoading(false);
  };

  useEffect(() => {
    fetchTests();

    // Subscribe to realtime changes
    const channel = supabase
      .channel('generated_tests_changes')
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'generated_tests'
        },
        () => {
          fetchTests();
        }
      )
      .subscribe();

    return () => {
      supabase.removeChannel(channel);
    };
  }, []);

  const handleApprove = async (testId: string) => {
    setProcessing(testId);
    const success = await TestGenerator.applyTest(testId);
    if (success) {
      await fetchTests();
    }
    setProcessing(null);
  };

  const handleReject = async (testId: string) => {
    setProcessing(testId);
    const success = await TestGenerator.rejectTest(testId);
    if (success) {
      await fetchTests();
    }
    setProcessing(null);
  };

  const getStatusBadge = (status: TestStatus) => {
    switch (status) {
      case 'approved':
        return (
          <Badge variant="default" className="gap-1">
            <CheckCircle2 className="w-3 h-3" />
            Approved
          </Badge>
        );
      case 'rejected':
        return (
          <Badge variant="secondary" className="gap-1">
            <XCircle className="w-3 h-3" />
            Rejected
          </Badge>
        );
      case 'pending':
        return (
          <Badge variant="outline" className="gap-1">
            <TestTube2 className="w-3 h-3" />
            Pending
          </Badge>
        );
      default:
        return null;
    }
  };

  if (loading) {
    return (
      <Card className="p-6">
        <div className="flex items-center gap-2 mb-4">
          <TestTube2 className="w-5 h-5 text-primary" />
          <h3 className="text-lg font-semibold">AI টেস্ট জেনারেটর • AI Test Generator</h3>
        </div>
        <p className="text-muted-foreground">লোড হচ্ছে • Loading...</p>
      </Card>
    );
  }

  return (
    <>
      <Card className="p-6">
        <div className="flex items-center justify-between mb-4">
          <div className="flex items-center gap-2">
            <TestTube2 className="w-5 h-5 text-primary" />
            <h3 className="text-lg font-semibold">AI টেস্ট জেনারেটর • AI Test Generator</h3>
            <Badge variant="secondary">{tests.length} pending</Badge>
          </div>
        </div>

        {tests.length === 0 ? (
          <p className="text-muted-foreground text-center py-8">
            কোনো অপেক্ষমাণ টেস্ট নেই • No pending tests
          </p>
        ) : (
          <ScrollArea className="h-[400px]">
            <div className="space-y-3">
              {tests.map((test) => (
                <Card key={test.id} className="p-4 bg-muted/30">
                  <div className="flex items-start justify-between gap-4">
                    <div className="flex-1 space-y-2">
                      <div className="flex items-center gap-2">
                        <FileCode className="w-4 h-4 text-primary" />
                        <code className="text-sm font-mono">{test.test_file_path}</code>
                        {getStatusBadge(test.status)}
                      </div>
                      
                      {test.description && (
                        <p className="text-sm text-muted-foreground">{test.description}</p>
                      )}

                      {test.scan_results && (
                        <div className="text-xs text-muted-foreground">
                          For error: {test.scan_results.message}
                        </div>
                      )}
                      
                      <div className="flex items-center gap-4 text-xs text-muted-foreground">
                        <div>Framework: {test.test_framework}</div>
                        <div>Created: {format(new Date(test.created_at), 'PPp')}</div>
                      </div>

                      <Button
                        size="sm"
                        variant="ghost"
                        onClick={() => setPreviewTest(test)}
                        className="gap-2"
                      >
                        <Eye className="w-4 h-4" />
                        টেস্ট কোড দেখুন • Preview Test
                      </Button>
                    </div>

                    {test.status === 'pending' && (
                      <div className="flex gap-2">
                        <Button
                          size="sm"
                          onClick={() => handleApprove(test.id)}
                          disabled={processing === test.id}
                          className="gap-2"
                        >
                          <CheckCircle2 className="w-4 h-4" />
                          {processing === test.id ? 'অনুমোদন করা হচ্ছে...' : 'Approve'}
                        </Button>
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => handleReject(test.id)}
                          disabled={processing === test.id}
                          className="gap-2"
                        >
                          <XCircle className="w-4 h-4" />
                          Reject
                        </Button>
                      </div>
                    )}
                  </div>
                </Card>
              ))}
            </div>
          </ScrollArea>
        )}
      </Card>

      {/* Preview Dialog */}
      <Dialog open={!!previewTest} onOpenChange={() => setPreviewTest(null)}>
        <DialogContent className="max-w-4xl max-h-[80vh]">
          <DialogHeader>
            <DialogTitle>টেস্ট কোড প্রিভিউ • Test Code Preview</DialogTitle>
            <DialogDescription>
              <code className="text-sm">{previewTest?.test_file_path}</code>
            </DialogDescription>
          </DialogHeader>
          
          <ScrollArea className="h-[500px]">
            <div className="space-y-4">
              {previewTest?.description && (
                <div>
                  <p className="font-semibold mb-2 text-sm">বর্ণনা • Description:</p>
                  <p className="text-sm text-muted-foreground">{previewTest.description}</p>
                </div>
              )}
              
              <div>
                <p className="font-semibold mb-2 text-sm">টেস্ট কোড • Test Code ({previewTest?.test_framework}):</p>
                <pre className="bg-muted p-4 rounded text-xs overflow-x-auto">
                  {previewTest?.test_code}
                </pre>
              </div>
            </div>
          </ScrollArea>

          <DialogFooter>
            <Button variant="outline" onClick={() => setPreviewTest(null)}>
              বন্ধ করুন • Close
            </Button>
            {previewTest && previewTest.status === 'pending' && (
              <>
                <Button
                  variant="outline"
                  onClick={() => {
                    handleReject(previewTest.id);
                    setPreviewTest(null);
                  }}
                  className="gap-2"
                >
                  <XCircle className="w-4 h-4" />
                  Reject
                </Button>
                <Button
                  onClick={() => {
                    handleApprove(previewTest.id);
                    setPreviewTest(null);
                  }}
                  className="gap-2"
                >
                  <CheckCircle2 className="w-4 h-4" />
                  Approve Test
                </Button>
              </>
            )}
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </>
  );
}
