import { useEffect, useState } from "react";
import { supabase } from "@/integrations/supabase/client";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Activity } from "lucide-react";
import type { Json } from "@/integrations/supabase/types";

type LogEntry = {
  id: string;
  type: string;
  message: string;
  details: Json | null;
  created_at: string;
  user_id?: string | null;
};

const LogStream = () => {
  const [logs, setLogs] = useState<LogEntry[]>([]);
  const [isStreaming, setIsStreaming] = useState(true);

  useEffect(() => {
    fetchInitialLogs();
    const channel = subscribeToLogs();
    
    return () => {
      supabase.removeChannel(channel);
    };
  }, []);

  const fetchInitialLogs = async () => {
    const { data } = await supabase
      .from('core_brain_logs')
      .select('*')
      .order('created_at', { ascending: false })
      .limit(50);

    if (data) {
      setLogs(data);
    }
  };

  const subscribeToLogs = () => {
    return supabase
      .channel('log_stream')
      .on(
        'postgres_changes',
        {
          event: 'INSERT',
          schema: 'public',
          table: 'core_brain_logs'
        },
        (payload) => {
          const newLog = payload.new as LogEntry;
          setLogs((prev) => [newLog, ...prev].slice(0, 50));
        }
      )
      .subscribe();
  };

  const getLogColor = (type: string) => {
    switch (type.toLowerCase()) {
      case 'error':
      case 'critical':
        return 'destructive';
      case 'warning':
        return 'secondary';
      case 'info':
        return 'default';
      case 'success':
        return 'outline';
      default:
        return 'default';
    }
  };

  const formatTime = (timestamp: string) => {
    const date = new Date(timestamp);
    return date.toLocaleTimeString('en-US', { 
      hour12: false, 
      hour: '2-digit', 
      minute: '2-digit', 
      second: '2-digit' 
    });
  };

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle className="flex items-center gap-2">
            <Activity className="w-5 h-5" />
            রিয়েলটাইম লগ স্ট্রিম • Real-time Log Stream
          </CardTitle>
          <div className="flex items-center gap-2">
            <div className={`w-2 h-2 rounded-full ${isStreaming ? 'bg-green-500 animate-pulse' : 'bg-gray-400'}`} />
            <span className="text-xs text-muted-foreground">
              {isStreaming ? 'সক্রিয় • Live' : 'বিরতিতে • Paused'}
            </span>
          </div>
        </div>
      </CardHeader>
      <CardContent>
        <ScrollArea className="h-[400px] pr-4">
          <div className="space-y-2">
            {logs.length === 0 ? (
              <div className="text-center py-8 text-muted-foreground">
                কোনো লগ পাওয়া যায়নি • No logs available
              </div>
            ) : (
              logs.map((log) => (
                <div
                  key={log.id}
                  className="flex items-start gap-3 p-3 rounded-lg bg-muted/50 hover:bg-muted transition-colors"
                >
                  <div className="flex-shrink-0 pt-0.5">
                    <Badge variant={getLogColor(log.type)} className="text-xs">
                      {log.type}
                    </Badge>
                  </div>
                  <div className="flex-1 min-w-0">
                    <div className="flex items-center gap-2 mb-1">
                      <span className="text-xs text-muted-foreground font-mono">
                        {formatTime(log.created_at)}
                      </span>
                    </div>
                    <p className="text-sm break-words">{log.message}</p>
                    {log.details && (
                      <pre className="text-xs text-muted-foreground mt-2 p-2 bg-background/50 rounded overflow-x-auto">
                        {JSON.stringify(log.details, null, 2)}
                      </pre>
                    )}
                  </div>
                </div>
              ))
            )}
          </div>
        </ScrollArea>
      </CardContent>
    </Card>
  );
};

export default LogStream;
