import { supabase } from "@/integrations/supabase/client";
import { toast } from "sonner";

export interface PatchData {
  fixPlanId: string;
  filePath: string;
  originalCode: string;
  patchedCode: string;
}

export class PatchEngine {
  /**
   * Generate a code patch from a fix plan
   */
  static async generatePatch(fixPlanId: string): Promise<string | null> {
    try {
      const { data: fixPlan } = await supabase
        .from('fix_plans')
        .select('*')
        .eq('id', fixPlanId)
        .single();

      if (!fixPlan) {
        throw new Error('Fix plan not found');
      }

      // Call the analyze function to generate code
      const { data, error } = await supabase.functions.invoke('core-brain-analyze', {
        body: {
          type: 'patch',
          context: {
            fix_plan_id: fixPlanId,
            title: fixPlan.title,
            description: fixPlan.description,
            ai_suggestion: fixPlan.ai_suggestion
          }
        }
      });

      if (error) throw error;

      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error('Not authenticated');

      // Save the patch
      const { data: patch, error: insertError } = await supabase
        .from('code_patches')
        .insert({
          fix_plan_id: fixPlanId,
          file_path: data.file_path || 'unknown',
          patched_code: data.code || '',
          original_code: null,
          status: 'pending',
          created_by: user.id
        })
        .select()
        .single();

      if (insertError) throw insertError;

      await supabase.from('core_brain_logs').insert({
        type: 'info',
        message: `Code patch generated for fix plan: ${fixPlan.title}`,
        details: { patch_id: patch.id, fix_plan_id: fixPlanId }
      });

      return patch.id;
    } catch (error) {
      console.error('Error generating patch:', error);
      toast.error('প্যাচ তৈরি ব্যর্থ • Failed to generate patch');
      return null;
    }
  }

  /**
   * Apply a code patch (marks as applied, actual file changes happen externally)
   */
  static async applyPatch(patchId: string): Promise<boolean> {
    try {
      const { data: patch } = await supabase
        .from('code_patches')
        .select('*')
        .eq('id', patchId)
        .single();

      if (!patch) {
        throw new Error('Patch not found');
      }

      if (patch.status === 'applied') {
        toast.info('এই প্যাচ ইতিমধ্যে প্রয়োগ করা হয়েছে • Patch already applied');
        return false;
      }

      // Mark patch as applied
      const { error: updateError } = await supabase
        .from('code_patches')
        .update({
          status: 'applied',
          applied_at: new Date().toISOString()
        })
        .eq('id', patchId);

      if (updateError) throw updateError;

      // Log the application
      await supabase.from('core_brain_logs').insert({
        type: 'success',
        message: `Code patch applied: ${patch.file_path}`,
        details: { patch_id: patchId, file_path: patch.file_path }
      });

      // Update related fix plan
      if (patch.fix_plan_id) {
        await supabase
          .from('fix_plans')
          .update({ status: 'completed' })
          .eq('id', patch.fix_plan_id);
      }

      toast.success('প্যাচ সফলভাবে প্রয়োগ! • Patch applied successfully!');
      return true;
    } catch (error) {
      console.error('Error applying patch:', error);
      toast.error('প্যাচ প্রয়োগ ব্যর্থ • Failed to apply patch');
      return false;
    }
  }

  /**
   * Reject/delete a code patch
   */
  static async rejectPatch(patchId: string): Promise<boolean> {
    try {
      const { error } = await supabase
        .from('code_patches')
        .delete()
        .eq('id', patchId);

      if (error) throw error;

      await supabase.from('core_brain_logs').insert({
        type: 'warning',
        message: `Code patch rejected and deleted`,
        details: { patch_id: patchId }
      });

      toast.info('প্যাচ বাতিল • Patch rejected');
      return true;
    } catch (error) {
      console.error('Error rejecting patch:', error);
      toast.error('প্যাচ বাতিল ব্যর্থ • Failed to reject patch');
      return false;
    }
  }

  /**
   * Rollback an applied code patch
   */
  static async rollbackPatch(patchId: string): Promise<boolean> {
    try {
      const { data: patch } = await supabase
        .from('code_patches')
        .select('*')
        .eq('id', patchId)
        .single();

      if (!patch) {
        throw new Error('Patch not found');
      }

      if (patch.status !== 'applied') {
        toast.error('শুধুমাত্র প্রয়োগকৃত প্যাচ রোলব্যাক করা যাবে • Only applied patches can be rolled back');
        return false;
      }

      if (!patch.original_code) {
        toast.error('মূল কোড পাওয়া যায়নি • Original code not available for rollback');
        return false;
      }

      // Mark patch as rolled back
      const { error: updateError } = await supabase
        .from('code_patches')
        .update({
          status: 'rolled_back',
          applied_at: null
        })
        .eq('id', patchId);

      if (updateError) throw updateError;

      // Log the rollback
      await supabase.from('core_brain_logs').insert({
        type: 'warning',
        message: `Code patch rolled back: ${patch.file_path}`,
        details: { patch_id: patchId, file_path: patch.file_path }
      });

      // Update related fix plan
      if (patch.fix_plan_id) {
        await supabase
          .from('fix_plans')
          .update({ status: 'pending' })
          .eq('id', patch.fix_plan_id);
      }

      toast.success('প্যাচ রোলব্যাক সফল! • Patch rolled back successfully!');
      return true;
    } catch (error) {
      console.error('Error rolling back patch:', error);
      toast.error('রোলব্যাক ব্যর্থ • Failed to rollback patch');
      return false;
    }
  }

  /**
   * Get all pending patches
   */
  static async getPendingPatches() {
    const { data, error } = await supabase
      .from('code_patches')
      .select('*')
      .eq('status', 'pending')
      .order('created_at', { ascending: false });

    if (error) {
      console.error('Error fetching pending patches:', error);
      return [];
    }

    return data || [];
  }

  /**
   * Get patch history (all patches)
   */
  static async getPatchHistory() {
    const { data, error } = await supabase
      .from('code_patches')
      .select('*')
      .order('created_at', { ascending: false });

    if (error) {
      console.error('Error fetching patch history:', error);
      return [];
    }

    return data || [];
  }
}
